#!/usr/bin/perl
#
# $Revision: 1.268 $
# Copyright 2009 Teleflora.
#
# Apply updates to the underlying Linux operating system.
# Primarily:
#	Apply redhat errata.
#	Register with redhat
#	(Un) Register with redhat.
#	Upgrade packages as needed for either RTI v14 or Daisy 8.
#

use strict;
use warnings;
use English;
use Getopt::Long;
use POSIX;
use File::Basename;
use Cwd;
use Sys::Hostname;
use Socket;


my $CVS_REVISION = '$Revision: 1.268 $';
my $TIMESTAMP = strftime("%Y%m%d%H%M%S", localtime());
my $PROGNAME = basename($0);

my $HELP = 0;
my $VERSION = 0;
my $VERBOSE = 0;
my $BAREMETAL = 0;
my $RTIV12 = 0;
my $RTIV14 = 0;
my $DAISYV8 = 0;
my $REMOUNT_TF = "";
my $OSPATCHES = 0;
my $RHNSYSTEMID = 0;
my $RHNREG = 0;
my $REBOOT = 0;
my $SWAPON = 0;
my $FORENSICS = 0;
my $OSTOOLS = 0;
my $UPS = 0;
my $UPS_SERIAL = 0;
my $UPS_SERIAL_PORT = "/dev/ttyS0";
my $MOTD = 0;
my $JAVA = 0;
my $JAVA_VERSION = "latest";
my $SAMBA = 0;
my $SAMBA_GEN_CONF = 0;
my $SAMBA_SET_PASSDB = 0;
my $SAMBA_REBUILD_PASSDB = 0;
my $INITTAB = 0;
my $ONETIMEROOT = 1;
my $IPADDR = "";
my $IPADDR_DEF = "192.168.1.21";
my $NETMASK = "";
my $NETMASK_DEF = "255.255.255.0";
my $GATEWAY = "";
my $GATEWAY_DEF = "192.168.1.1";
my $IF_NAME = "";
my $IF_NAME_DEF = "eth0";
my $NAMESERVER = "";
my $NAMESERVER_DEF = "8.8.8.8";
my $HOSTNAME = "";
my $TFSERVER = "rtihardware.homelinux.com";
my $KEEPDHCP = "";
my $KEEPKERNELS = 8;
my $YUMCONF = 0;
my $UPDATEDB = 0;
my $INIT_CONSOLE_RES = 0;
my $ENABLE_BOOT_MSGS = 0;
my $DISABLE_KMS = 0;
my $SYSLOG_MARK = -1;
my $KLOG_MSG_PRIORITY = -1;
my $KLOG_MSG_PRIORITY_DEF = 3;
my $KERNEL_MSG_CONSOLE = "";
my $UNINSTALL_READAHEAD = 0;
my $DRY_RUN = 0;

my $TEST_CONFIG_FILE_PATH = "";
my $TEST_SYSCONFIG_HOSTNAME_CHANGE = 0;
my $TEST_HOSTS_CONFIG_FILE_CHANGE = 0;

my $CUPSTMP = 0;
my $CUPSCONF = 0;
my $CUPSCONF_MAXJOBS = 1;
my $CUPSCONF_DISABLETEMPFILES = 1;
my $CUPSCONF_TIMEOUTS = 1;
my $CUPSCONF_ERRORPOLICY = 1;
my $CUPSCONF_RHEL6_TIMEOUT = 300;
my $CUPSCONF_RHEL5_TIMEOUT = 0;
my $PURGEPRINT = 0;
my $PURGERPMS = 0;
my $OS = "";


#
# Alternate os and root for testing.
#
my $ALTROOT = "";
my $ALTOS = "";

#
# Values for $ARCH: "", "i386" (32 bit), "x86_64" (64 bit)
#
my $ARCH = "";

#
# Values for $POSDIR: "", "/d/daisy", "/usr2/bbx"
#
my $POSDIR = "";

#
# $DAISY and $RTI are 0 (FALSE) or 1 (TRUE)
#
my $DAISY = 0;
my $RTI = 0;

#
# Path to the logfile - it will either be in the POS tree or if no POS
# has yet been installed, then the log file will be in /tmp.
#
my $LOGFILE_PATH = "";

#
# The rev numbers are "magic" and this script has to be edited and a new
# release of ostools produced if they change - less than desirable.
#
my $BBJ = "2145830.jar.gz";
my $BLM = "blm.2145830.jar.gz";

#
# Used for fork of aide database rebuild
#
my $PID = 0;

#
# Constants
#

my $EMPTY_STR = q{};

# Exit status values
my $EXIT_OK = 0;
my $EXIT_COMMAND_LINE = 1;
my $EXIT_MUST_BE_ROOT = 2;
my $EXIT_SAMBA_CONF = 3;
my $EXIT_GRUB_CONF = 4;
my $EXIT_ARCH = 5;
my $EXIT_JAVA_VERSION = 10;
my $EXIT_JAVA_DOWNLOAD = 11;
my $EXIT_JAVA_INSTALL = 12;
my $EXIT_RTI14 = 14;
my $EXIT_READAHEAD = 15;
my $EXIT_OSPATCHES = 20;
my $EXIT_WRONG_PLATFORM = 21;
my $EXIT_RHWS_CONVERT = 22;
my $EXIT_UP2DATE = 23;
my $EXIT_YUM_UPDATE = 24;
my $EXIT_DIGI_DRIVERS = 25;
my $EXIT_INITSCRIPTS = 26;
my $EXIT_RHN_NOT_REGISTERED = 27;
my $EXIT_HOSTNAME_CHANGE = 30;

# POS type
my $POS_TYPE_RTI = "rti";
my $POS_TYPE_DAISY = "daisy";

# Network attribute selector
my $NET_ATTR_IPADDR = 1;
my $NET_ATTR_BROADCAST = 2;
my $NET_ATTR_NETMASK = 3;


GetOptions(
	"help" => \$HELP,
	"version" => \$VERSION,
	"verbose" => \$VERBOSE,
	"baremetal" => \$BAREMETAL,
	"rti12" => \$RTIV12,
	"rti14" => \$RTIV14,
	"daisy8" => \$DAISYV8,
	"remount=s" => \$REMOUNT_TF,
	"cupstmp" => \$CUPSTMP,
	"cupsconf" => \$CUPSCONF,
	"purgeprint" => \$PURGEPRINT,
	"purgerpms" => \$PURGERPMS,
	"ups" => \$UPS,
	"ups-serial" => \$UPS_SERIAL,
	"ups-serial-port=s" => \$UPS_SERIAL_PORT,
	"motd" => \$MOTD,
	"java" => \$JAVA,
	"java-version=s" => \$JAVA_VERSION,
	"samba" => \$SAMBA,
	"samba-gen-conf" => \$SAMBA_GEN_CONF,
	"samba-set-passdb" => \$SAMBA_SET_PASSDB,
	"samba-rebuild-passdb" => \$SAMBA_REBUILD_PASSDB,
	"inittab" => \$INITTAB,
	"ospatches" => \$OSPATCHES,
	"rhnsystemid" => \$RHNSYSTEMID,
	"ostools" => \$OSTOOLS,
	"rhnreg" => \$RHNREG,
	"hardboot" => \$REBOOT,
	"reboot" => \$REBOOT,
	"swapon" => \$SWAPON,
	"forensics" => \$FORENSICS,
	"onetimeroot" => \$ONETIMEROOT,
	"ipaddr=s" => \$IPADDR,
	"netmask=s" => \$NETMASK,
	"gateway=s" => \$GATEWAY,
	"ifname=s" => \$IF_NAME,
	"nameserver=s" => \$NAMESERVER,
	"hostname=s" => \$HOSTNAME,
	"tfserver=s" => \$TFSERVER,
	"keepdhcp" => \$KEEPDHCP,
	"keepkernels=s" => \$KEEPKERNELS,
	"yumconf" => \$YUMCONF,
	"updatedb" => \$UPDATEDB,
	"init-console-res" => \$INIT_CONSOLE_RES,
	"enable-boot-msgs" => \$ENABLE_BOOT_MSGS,
	"disable-kms" => \$DISABLE_KMS,
	"syslog-mark=s" => \$SYSLOG_MARK,
	"klog-msg-priority=s" => \$KLOG_MSG_PRIORITY,
	"kernel-msg-console=s" => \$KERNEL_MSG_CONSOLE,
	"uninstall-readahead" => \$UNINSTALL_READAHEAD,
	"altroot=s" => \$ALTROOT,
	"altos=s" => \$ALTOS,
	"dry-run" => \$DRY_RUN,
	"test-config-file-path=s" => \$TEST_CONFIG_FILE_PATH,
	"test-sysconfig-hostname-change" => \$TEST_SYSCONFIG_HOSTNAME_CHANGE,
	"test-hosts-config-file-change" => \$TEST_HOSTS_CONFIG_FILE_CHANGE,
) || die "Error: invalid command line option, exiting...\n";



# --version
if($VERSION != 0) {
	print("OSTools Version: 1.14.0\n");
	print("$PROGNAME $CVS_REVISION\n");
	exit(0);
}

# --help
if($HELP != 0) {
	usage();
	exit(0);
}


if (-d '/d/daisy') {
	$POSDIR = '/d/daisy';
	$DAISY = 1;
} elsif (-d '/usr2/bbx') {
	$POSDIR = '/usr2/bbx';
	$RTI = 1;
} else {
	$POSDIR = "";
	showinfo("Can not detect that a POS has been installed - assuming none.");
}


$OS = determine_os();

$ARCH = processor_arch();


# We must be root to do these things.
unless (is_running_in_test_mode()) {
    if ($ALTROOT eq "" && $EUID != 0) {
	showinfo("$PROGNAME must be run as root or with sudo");
	exit($EXIT_MUST_BE_ROOT);
    }
}

#########################################
######### TEST OPTIONS PARSER ###########
#########################################

# --test-sysconfig-hostname-change
if ($TEST_SYSCONFIG_HOSTNAME_CHANGE) {
    if ($TEST_CONFIG_FILE_PATH eq $EMPTY_STR) {
	$TEST_CONFIG_FILE_PATH = "/etc/sysconfig/network";
    }
    unless (is_arg_ok($TEST_CONFIG_FILE_PATH)) {
	showerror("value of --test-config-file-path insecure: $TEST_CONFIG_FILE_PATH");
	exit($EXIT_COMMAND_LINE);
    }
    unless (-e $TEST_CONFIG_FILE_PATH) {
	showerror("config file does not exist: $TEST_CONFIG_FILE_PATH");
	exit($EXIT_COMMAND_LINE);
    }
    if ($HOSTNAME eq $EMPTY_STR) {
	showerror("value for --hostname must be specified");
	exit($EXIT_COMMAND_LINE);
    }
    unless (is_arg_ok($HOSTNAME)) {
	showerror("value of --hostname insecure: $HOSTNAME");
	exit($EXIT_COMMAND_LINE);
    }

    exit(test_sysconfig_hostname_change($HOSTNAME, $TEST_CONFIG_FILE_PATH));
}

# --test-hosts-config-file-change
if ($TEST_HOSTS_CONFIG_FILE_CHANGE) {
    if ($TEST_CONFIG_FILE_PATH eq $EMPTY_STR) {
	$TEST_CONFIG_FILE_PATH = "/etc/hosts";
    }
    unless (is_arg_ok($TEST_CONFIG_FILE_PATH)) {
	showerror("value of --test-config-file-path insecure: $TEST_CONFIG_FILE_PATH");
	exit($EXIT_COMMAND_LINE);
    }
    unless (-e $TEST_CONFIG_FILE_PATH) {
	showerror("config file does not exist: $TEST_CONFIG_FILE_PATH");
	exit($EXIT_COMMAND_LINE);
    }
    if ($HOSTNAME eq $EMPTY_STR) {
	showerror("value for --hostname must be specified");
	exit($EXIT_COMMAND_LINE);
    }
    unless (is_arg_ok($HOSTNAME)) {
	showerror("value of --hostname insecure: $HOSTNAME");
	exit($EXIT_COMMAND_LINE);
    }

    exit(test_hosts_config_file_change($HOSTNAME, $TEST_CONFIG_FILE_PATH));
}

# PCI 6.5.6
#
# Look for a list of patterns in an input string that might
# indicate that some insecure value was passed into the script.
# Example, no input scring should have the BACKQUOTE chars as
# part of the string... if there were, it could mean that
# someone was trying to get the script to run another unknown
# script as "root".
#
# Returns
#   1 if arg is OK
#   0 if arg is insecure
#
sub is_arg_ok
{
    my ($arg) = @_;

    return(1) if ($arg eq "");

    my $rc = 1;

    my @insecure_patterns = (
	'\`',               # `bad command`
	'(\$\()(.*.)(\))',  # $(bad command)
	'\;',               # stuff ; bad command
	'\&',               # stuff && bad command
	'\|',               # stuff | bad command
	'\>',               # stuff > bad command
	'\<',               # stuff < bad command
	'[[:cntrl:]]',      # non printables
    );

    foreach my $re (@insecure_patterns) {
	if ($arg =~ /$re/) {
	    $rc = 0;
	    last;
	}
    }

    return($rc);
}


###########################################
######### PUBLIC OPTIONS PARSER ###########
###########################################

	showinfo("Starting $PROGNAME $CVS_REVISION...");

	# --rhnreg
	if ($RHNREG) {
	    my $rhn_system_id = get_rhn_system_id();
	    if ($rhn_system_id) {
		showinfo("system already registered with Red Hat Network: $rhn_system_id");
		exit($EXIT_OK);
	    }
	    else {
		showinfo("Obtaining a Redhat License...");
		$rhn_system_id = register_redhat();
		if ($rhn_system_id) {
		    showinfo("Red Hat Network system id: $rhn_system_id");
		    exit($EXIT_OK);
		}
		else {
		    showerror("could not register system with Red Hat Network");
		    exit($EXIT_RHN_NOT_REGISTERED);
		}
	    }
	}

	# --baremetal
	if($BAREMETAL != 0) {
		showinfo("Using Teleflora Package Server: \"$TFSERVER\"");
		exit(baremetal_install());
	}

	# --remount=s
	if ($REMOUNT_TF ne "") {
		if ( ($REMOUNT_TF ne "/usr2") && ($REMOUNT_TF ne "/d") ) {
		    showerror("The /teleflora remount option arg must be one of /usr2 or /d");
		    exit(1);
		}
		exit(remount_filesystem("/teleflora", $REMOUNT_TF));
	}

	# --ospatches
	if ($OSPATCHES != 0) {
		exit(update_ospatches());
	}

	# --rhnsystemid
	if($RHNSYSTEMID != 0) {
		exit(report_rhn_system_id());
	}

	# --ostools
	if($OSTOOLS != 0) {
		exit(update_ostools());
	}

	# --init-console-res
	if ($INIT_CONSOLE_RES != 0) {
		exit(init_console_res());
	}

	# --enable-boot-msgs
	if ($ENABLE_BOOT_MSGS != 0) {
		exit(enable_boot_msgs());
	}

	# --disable-kms
	if ($DISABLE_KMS != 0) {
		exit(disable_kms());
	}

	# --uninstall-readahead
	if ($UNINSTALL_READAHEAD) {
		exit(uninstall_readahead());
	}

	# --syslog-mark
	if ($SYSLOG_MARK != -1) {
		exit(modify_syslog_mark($SYSLOG_MARK));
	}

	# --klog-msg-priority
	if ($KLOG_MSG_PRIORITY != -1) {
		exit(modify_klog_msg_priority($KLOG_MSG_PRIORITY));
	}

	# --kernel-msg-console
	if ($KERNEL_MSG_CONSOLE) {
		exit(set_kernel_msg_console($KERNEL_MSG_CONSOLE));
	}


	# --netmask=x.x.x.x
	if ( $NETMASK && ($IPADDR eq "") ) {
	    showerror("The --netmask option not allowed unless --ipaddr also specified");
	    exit(1);
	}

	# --gateway=x.x.x.x
	if ( $GATEWAY && ($IPADDR eq "") ) {
	    showerror("The --gateway option not allowed unless --ipaddr also specified");
	    exit(1);
	}

	# --ifname=string
	if ($IF_NAME) {
	    if ($IPADDR eq "") {
		showerror("The --ifname option not allowed unless --ipaddr also specified");
		exit(1);
	    }
	}
	else {
	    # if --ifname not specified, use the default value.
	    $IF_NAME = $IF_NAME_DEF;
	}


	# --rti14
	if ($RTIV14 != 0) {
	    exit(install_rtiv14_packages($IF_NAME));
	}

	# --rti12
	if ($RTIV12 != 0) {
	    exit(install_rtiv12_packages($IF_NAME));
	}

	# --daisy8
	if ($DAISYV8 != 0) {
	    exit(install_daisyv8_packages($IF_NAME));
	}

	# --ipaddr=x.x.x.x
	if ($IPADDR) {
	    if ($IPADDR ne $IPADDR_DEF && ($NETMASK eq "" || $GATEWAY eq "")) {
		showerror("If ip addr not $IPADDR_DEF, --netmask and --gateway must also be specified");
		exit(1);
	    }

	    if (configure_ip_addr($IPADDR, $NETMASK, $GATEWAY, $IF_NAME) != -1) {
		if (update_ipaddr($HOSTNAME, $IPADDR) != -1) {

		    # if --hostname not specified, then we are done
		    if ($HOSTNAME eq "") {
			exit(0);
		    }
		}
		else {
		    exit(1);
		}
	    }
	    else {
		exit(1);
	    }

	    # if we get here, do the hostname option if specified
	}

	# --hostname=name
	if ($HOSTNAME) {
	    exit(update_hostname($HOSTNAME, $IPADDR, $IF_NAME));
	}

	# --nameserver=x.x.x.x
	if ($NAMESERVER) {
	    exit(update_nameserver($NAMESERVER));
	}

	# --cupstmp
	if ($CUPSTMP != 0) {
		cups_clean_tempfiles();
	}

	# --cupsconf
	if ($CUPSCONF != 0) {
		cups_edit_conf();
	}

	# --purgeprint
	if($PURGEPRINT != 0) {
		purge_printjobs();
	}

	# --purgerpms
	if($PURGERPMS != 0) {
		purge_rpms();
	}

	# --ups
	if ($UPS || $UPS_SERIAL) {
		exit(install_apcupsd());
	}

	# --motd
	if($MOTD != 0) {
		exit(modify_motd());
	}

	# --java
	if ($JAVA != 0) {
	    exit(install_java($JAVA_VERSION));
	}

	# --samba
	# --samba-gen-conf
	if ( ($SAMBA != 0) || ($SAMBA_GEN_CONF != 0) ) {
		my $pos_type = "";
		if ($RTI) {
		    $pos_type = $POS_TYPE_RTI;
		}
		elsif ($DAISY) {
		    $pos_type = $POS_TYPE_DAISY;
		}
		exit(modify_samba($pos_type));
	}

	# --samba-set-passdb
	if ($SAMBA_SET_PASSDB != 0) {
		exit(samba_set_passdb());
	}

	# --samba-rebuild-passdb
	if ($SAMBA_REBUILD_PASSDB != 0) {
		exit(samba_rebuild_passdb());
	}

	# --inittab
	if($INITTAB != 0) {
		exit(modify_inittab());
	}

	# --updatedb
	if($UPDATEDB != 0) {
		exit(modify_updatedb());
	}

	# --hardboot
	# --reboot
	if($REBOOT != 0) {
		exit(hard_reboot());
	}

	# --swapon
	if($SWAPON != 0) {
		exit(swapon());
	}

	# --forensics
	if($FORENSICS != 0) {
		exit(forensics());
	}

	# --yumconf
	if ($YUMCONF) {
		update_yum_config();
	}


exit(0);
###################################################
###################################################
###################################################


sub usage
{
	print(<< "EOF");
$PROGNAME Usage:

$PROGNAME --version
$PROGNAME --help
$PROGNAME --verbose
$PROGNAME --baremetal
$PROGNAME --rti12 [--tfserver=fqdn] [--ipaddr=x.x.x.x]
$PROGNAME --rti14 [--tfserver=fqdn] [--ipaddr=x.x.x.x]
$PROGNAME --daisy8 [--tfserver=fqdn] [--ipaddr=x.x.x.x] [--keepdhcp]
$PROGNAME --ostools
$PROGNAME --reboot

-- Defaults --
    --tfserver=rtihardware.homelinux.com
    --ipaddr=192.168.1.21
    --netmask=255.255.255.0
    --gateway=192.168.1.1
    --ifname=eth0
    --keepkernels=6

-- Advanced Options --
$PROGNAME --ipaddr=x.x.x.x [--netmask=x.x.x.x] [--gateway=x.x.x.x] [--ifname=name]
$PROGNAME --hostname=name
$PROGNAME --nameserver=x.x.x.x
$PROGNAME --remount=/usr2       # remount /teleflora as /usr2
$PROGNAME --remount=/d          # remount /teleflora as /d
$PROGNAME --keepkernels=n
$PROGNAME --syslog-mark=n
$PROGNAME --klog-msg-priority=n
$PROGNAME --kernel-msg-console=/dev/ttyn
$PROGNAME --init-console-res
$PROGNAME --enable-boot-msgs
$PROGNAME --disable-kms
$PROGNAME --uninstall-readahead
$PROGNAME --java
$PROGNAME --cupstmp
$PROGNAME --cupsconf
$PROGNAME --purgeprint
$PROGNAME --purgerpms
$PROGNAME --ups
$PROGNAME --ups-serial
$PROGNAME --ups-serial-port=/dev/ttySn 
$PROGNAME --motd
$PROGNAME --samba
$PROGNAME --samba-gen-conf         # generate appropriate smb.conf file
$PROGNAME --samba-set-passdb       # configure samba to use smbpasswd
$PROGNAME --samba-rebuild-passdb   # rebuild samba smbpasswd file
$PROGNAME --ospatches
$PROGNAME --rhnsystemid
$PROGNAME --inittab
$PROGNAME --rhnreg
$PROGNAME --swapon
$PROGNAME --yumconf
$PROGNAME --updatedb
$PROGNAME --forensics > /tmp/forensics.txt
EOF

}


###################################
####### TOP LEVEL TEST SUBS #######
###################################

sub is_running_in_test_mode
{
    my $rc = 0;

    if ($TEST_SYSCONFIG_HOSTNAME_CHANGE) {
	$rc = 1;
    }
    if ($TEST_HOSTS_CONFIG_FILE_CHANGE) {
	$rc = 1;
    }

    return($rc);
}


sub test_sysconfig_hostname_change
{
    my ($hostname, $config_file_path) = @_;

    my $rc = $EXIT_OK;

    if (set_sysconfig_hostname($hostname, $config_file_path)) {
	showinfo("HOSTNAME set to $hostname in: $config_file_path");
    }
    else {
	logerror("could not set HOSTNAME in config file: $config_file_path");
	$rc = $EXIT_HOSTNAME_CHANGE;
    }

    return($rc);
}


sub test_hosts_config_file_change
{
    my ($hostname, $config_file_path) = @_;

    my $rc = $EXIT_OK;

    my $ipaddr = get_network_attribute("eth0", $NET_ATTR_IPADDR);

    if (update_hosts_config_file($hostname, $ipaddr, $config_file_path)) {
	showinfo("hostname in $config_file_path changed to: $hostname");
    }
    else {
	logerror("could not change hostname in config file: $config_file_path");
	$rc = $EXIT_HOSTNAME_CHANGE;
    }

    return($rc);
}


# Register with the redhat portal.
# Make sure our rhn daemon is enabled.
sub register_redhat
{
	my $svctag = "";
	my $vendor = "";
	my $hardware = "";

	my $rhn_system_id = "";

	# get name of manufacturer and
	# get service tag number and
	# get the product name
	my $dmi_cmd = '/usr/sbin/dmidecode';
	if (open(my $pipe, '-|', "$dmi_cmd | grep -A 5 \"System Information\"")) {
	    while(<$pipe>) {
		chomp;
		if(/Manufacturer:\s+(\S+)/) {
			$vendor = $1;
		}
		if(/Serial Number:\s+(\S+)/) {
			$svctag = uc($1);
		}
		if(/Product Name:\s+([[:print:]]+)$/) {
			$hardware = $1;
		}

	    }
	    close($pipe);
	}
	else {
	    logerror("could not get system hardware info via: $dmi_cmd");
	    return($rhn_system_id);
	}


	my $profile_name = "--profilename \"$svctag $vendor $hardware\"";
	my $activation_key = "--activationkey=0ad77379739f3e7b6c3263070a1ab0dc";

	system("/usr/sbin/rhnreg_ks --force $profile_name $activation_key");


	foreach my $service_name ("rhnsd", "yum-updatesd") {
		if (-f "/etc/rc.d/init.d/$service_name") {
			system("/sbin/chkconfig --level 3 $service_name on");
			system("/sbin/chkconfig --level 5 $service_name on");
		}
	}

	#
	# Grab and print system ID here.
	# Verify that the Redhat Network Registration succeeded.
	# And let the user know.
	#
	$rhn_system_id = get_rhn_system_id();

	return($rhn_system_id);
}


#
# Sort of a "post install" set of steps for a new server.
#
sub baremetal_install
{
	system("grep Cent /etc/redhat-release > /dev/null 2> /dev/null");
	if ($? == 0) {
	    showinfo("CentOS does not need to register.");
	}
	else {
	    showinfo("Obtaining a Redhat License...");

	    my $rhn_system_id = register_redhat();
	    if ($rhn_system_id) {
		showinfo("Red Hat Network system id: $rhn_system_id");
	    }
	    else {
		showerror("could not register system with Red Hat Network");
		return($EXIT_RHN_NOT_REGISTERED);
	    }
	}


	remount_filesystem("/", "/");

	remount_swap();

	showinfo("Configuring System...");

	# if nmap is not installed, install it now
	system("rpm -qa | grep -q nmap");
	if ($? != 0) {
	    system("yum install -y nmap");
	}

	modify_inittab();
	modify_updatedb();
	modify_motd();
	cups_edit_conf();

	#
	# Set the kernel log message priority threshold - only messages
	# at the threshold or of greater priority are allowed to appear
	# on the console.  The priority will either be the value specified
	# on the command line or a default value.
	#
	my $klog_msg_priority = ($KLOG_MSG_PRIORITY != -1) ?
	    $KLOG_MSG_PRIORITY : $KLOG_MSG_PRIORITY_DEF;

	modify_klog_msg_priority($klog_msg_priority);

	#
	# For RHEL6 systems:
	#   Edit grub.conf:
	#   1) init the console rez
	#   2) enable verbose boot msgs by removing "rhgb" and "quiet"
	#   3) disable kernel (video) mode setting
	#   Also:
	#   1) uninstall readahead rpm
	#
	if ($OS eq "RHEL6") {
	    init_console_res();
	    enable_boot_msgs();
	    disable_kms();
	    uninstall_readahead();
	}

	if ($OS eq "RHEL4" || $OS eq "RHEL5" || $OS eq "RHEL6") {
	    install_apcupsd();
	}

	return($EXIT_OK);
}


#
# Modify system config file (default): /etc/sysconfig/network-scripts/ifcfg-eth0
#
# For all platforms, change:
#   BOOTPROTO="dhcp" --> BOOTPROTO="static"
# For all platforms, add
#   IPADDR="$ipaddr"
#   NETMASK="$netmask"
#   GATEWAY="$gateway"
# For RHEL6, add
#   NM_CONTROLLED="no"
#
# Return -1 for error, 0 for success
#
sub configure_ip_addr
{
	my $ipaddr = $_[0];
	my $netmask = $_[1];
	my $gateway = $_[2];
	my $if_name = $_[3];
	my $conf_file = "$ALTROOT/etc/sysconfig/network-scripts/ifcfg-$if_name";
	my $new_conf_file = "$conf_file.$$";

	showinfo("Setting Host IP Address (eth0) to \"$ipaddr\"...");
	if ($ipaddr !~ /^([\d]+)\.([\d]+)\.([\d]+)\.([\d]+)$/) {
		showerror("invalid: $ipaddr");
		return(-1);
	}

	unless (-f "$conf_file") {
		showerror("$conf_file does not exist.");
		return(-1);
	}

	unless (open(OLD, "< $conf_file")) {
		showerror("Could not open $conf_file for read.");
		return(-1);
	}

	unless (open(NEW, "> $new_conf_file")) {
		showerror("Could not open $new_conf_file for write.");
		close(OLD);
		return(-1);
	}

	my $is_nm_already_disabled = 0;
	while (<OLD>) {
		if ( (/^BOOTPROTO=dhcp/) || (/^BOOTPROTO="dhcp"/) ) {
			print(NEW "BOOTPROTO=\"static\"\n");
		}
		elsif ($OS eq "RHEL6") {
			if ( (/^NM_CONTROLLED=yes/) || (/^NM_CONTROLLED="yes"/) ) {
				print(NEW "NM_CONTROLLED=\"no\"\n");
				$is_nm_already_disabled = 1;
				next;
			}
			elsif ( (/^NM_CONTROLLED=no/) || (/^NM_CONTROLLED="no"/) ) {
				$is_nm_already_disabled = 1;
			}

			print(NEW $_);
		}
		elsif (/^IPADDR=/) {
			next;
		}
		elsif (/^GATEWAY=/) {
			next;
		}
		elsif (/^NETMASK=/) {
			next;
		}
		elsif (/^# Configuration generated by $PROGNAME/) {
			next;
		}
		else {
			# By default, just copy what we read.
			print(NEW $_);
		}
	}

	close(OLD);
	close(NEW);

	# It's an error if the new conf file is zero length at this point.
	if (-z $new_conf_file) {
		showerror("new conf file is zero length.");
		system("rm $new_conf_file");
		return(-1);
	}

	# Open new conf file again to append some more lines.
	unless (open(NEW, ">> $new_conf_file")) {
		showerror("Could not open $new_conf_file for append.");
		close(OLD);
		return(-1);
	}
	print(NEW "# Configuration generated by $PROGNAME $CVS_REVISION $TIMESTAMP\n");
	print(NEW << "EOF");
IPADDR=\"$ipaddr\"
NETMASK=\"$netmask\"
GATEWAY=\"$gateway\"
EOF
	if ( ($OS eq "RHEL6") && ($is_nm_already_disabled != 1) ) {
	    print(NEW "NM_CONTROLLED=\"no\"\n");
	}

	close(NEW);

	# Assume conf file was successfully transformed... so replace the old one with the new.
	system("chmod --reference=$conf_file $new_conf_file");
	system("chown --reference=$conf_file $new_conf_file");
	system("mv $new_conf_file $conf_file");

	print("Reboot or run \"# service network restart\" to activate network changes.\n");

	return(0);
}

#
# Return -1 for error, 0 for success
#

sub update_ipaddr
{
	my $hostname = $_[0];
	my $ipaddr = $_[1];
	my $fqdn = "";
	my $domain = "teleflora.com";

	if ($hostname eq "") {
	    $hostname = get_hostname();
	    if ($hostname eq "") {
		showerror("Can not get hostname or hostname not set.");
		showerror("The ipaddr in /etc/hosts will not be updated.");
		return(-1);
	    }
	}

        # if fully qualified domain name present, remove after saving it
        if ($hostname =~ /(.+)\.$domain/) {
            $fqdn = $hostname;
            $hostname =~ s/\.$domain//;
        }

        # if domain not present, form a fully qualified (if bogus) domain name
        else {
            $fqdn = $hostname . ".$domain";
        }

	# form a new line suitable for the /etc/hosts file
	my $hosts_file_entry = "$ipaddr\t$fqdn $hostname\n";

	# edit hosts file.
        open(OLDFILE, "< $ALTROOT/etc/hosts");
        open(NEWFILE, "> $ALTROOT/etc/hosts.$$");
        while (<OLDFILE>) {

	    # pass through blank lines and comment lines
	    if (/^$/ || /^(\s*)#/) {
		print(NEWFILE);
		next;
	    }

	    # pass through the loopback line
	    elsif (/^127\.0\.0\.1/) {
		print(NEWFILE);
		next;
	    }

	    # for non loopback lines:
	    else {
		if (/$hostname/) {
		    $_ = $hosts_file_entry;
		}
	    }

	    print(NEWFILE);
        }
        close(OLDFILE);
        close(NEWFILE);

	system("mv $ALTROOT/etc/hosts.$$ $ALTROOT/etc/hosts");

	return(0);
}


#
# On Redhat systems, the steps for changing the hostname are:
# 1) edit /etc/sysconfig/network
# 2) edit /etc/hosts
# 3) run the hostname(1) command
# 4) reboot or at least restart the network service
#
# Four possible cases with respect to contents of /etc/hosts:
# 1) the hostname does not appear
# 2) the hostname is only on the localhost line
# 3) the hostname is only on a line by itself
# 4) the hostname is on both the localhost line and a line by itself
#
# Returns:
#   $EXIT_OK on success
#   $EXIT_HOSTNAME_CHANGE on error
#
sub update_hostname
{
    my ($new_hostname, $ipaddr, $device) = @_;

    # just a warning
    if ($DAISY) {
	unless ($new_hostname =~ /^\d{8}-tsrvr/) {
	    showerror("Daisy hostnames should be of the form: shopcode-tsrvr");
	    showerror("Eg: for shopcode of \"12345600\", hostname would be: 12345600-tsrvr");
	}
    }

    # if IP address not changing, use current IP address
    if ($ipaddr eq "") {
	my $current_ipaddr = get_network_attribute($device, $NET_ATTR_IPADDR);
	if ($current_ipaddr eq "") {
	    showerror("could not get current IP address");
	    return($EXIT_HOSTNAME_CHANGE);
	}
	$ipaddr = $current_ipaddr;
    }


    #
    # step 1: edit the sysconfig network file
    #
    $new_hostname = strip_domain($new_hostname, "teleflora.com");
    my $config_file_path = "/etc/sysconfig/network";
    if (set_sysconfig_hostname($new_hostname, $config_file_path)) {
	showinfo("HOSTNAME variable in $config_file_path changed to: $new_hostname");
    }
    else {
	logerror("could not change HOSTNAME variable in config file: $config_file_path");
	return($EXIT_HOSTNAME_CHANGE);
    }

    #
    # step 2: edit the hosts file
    #

    $config_file_path = "/etc/hosts";
    if (update_hosts_config_file($new_hostname, $ipaddr, $config_file_path)) {
	showinfo("hostname in $config_file_path changed to: $new_hostname");
    }
    else {
	logerror("could not change hostname in config file: $config_file_path");
	return($EXIT_HOSTNAME_CHANGE);
    }

    #
    # step 3: run the hostname(1) command
    #
    system("hostname $new_hostname");
    if ($? == 0) {
	showinfo("hostname changed via command: hostname(1)");
    }
    else {
	showerror("hostname(1) command returned non-zero exit status: $?");
	return($EXIT_HOSTNAME_CHANGE);
    }

    showinfo("Reboot system or restart network to complete hostname change to $new_hostname");

    return($EXIT_OK);
}


#
# Return 0 on success, non-zero on error
#
sub update_nameserver
{
    my $nameserver = $_[0];
    my $conf_file = "$ALTROOT/etc/resolv.conf";

    if ($nameserver eq "") {
	showerror("Specified DNS name server ip addr is a null string");
	$nameserver = $NAMESERVER_DEF;
    }

    if ($nameserver !~ /^([\d]+)\.([\d]+)\.([\d]+)\.([\d]+)$/) {
	showerror("Specified DNS name server ip addr invalid: $nameserver");
	$nameserver = $NAMESERVER_DEF;
    }

    showinfo("Setting $nameserver as the ip addr of the DNS name server");

    open(FILE, "> $conf_file");

    print(FILE "; generated by $PROGNAME $CVS_REVISION $TIMESTAMP\n");
    print(FILE "nameserver $nameserver\n");

    close(FILE);

    return(0);
}


#
# Return value of function is used as exit status of script when
# called via the "--ospatches" command line option.
#
# Return    Description
# =====================
#   1       The operating system was not RHEL4, RHEL5, RHEL6
#   2       The operating system was RHWS and it could not be converted to server
#   3       The up2date command on RHEL4 failed
#   4       The yum command failed on RHEL{5,6} failed

sub update_ospatches
{
	unless ( ($OS eq "RHEL6") || ($OS eq "RHEL4") || ($OS eq "RHEL5") || ($OS eq "RHWS5") ) {
	    return($EXIT_WRONG_PLATFORM);
	}

	showinfo("Begin Installing OS Patches...");

	my $timestamp = "";
	$timestamp = strftime("%Y%m%d%H%M%S", localtime());
	showinfo("Timestamp: $timestamp");

	# Redhat Enterprise Client Workstation -> Redhat Enterprise Server Update
	# This to keep our redhat licensing up-to-snuff.
	# Note this process is per an email from Ian at Redhat.
	system("rpm -qa | grep redhat-release | grep 'redhat-release-5Client' > /dev/null 2> /dev/null");
	if ($? == 0) {

	    showinfo("Converting RHEL Client Workstation to RHELinux Server...\n");

	    # Replace RPMs.
	    my $pkg_name = "redhat-release-5Server-5.5.0.2.i386.rpm";
	    if ($ARCH eq "x86_64") {
		$pkg_name = "redhat-release-5Server-5.5.0.2.x86_64.rpm";
	    }
	    system("rpm -Uvh --oldpackage http://$TFSERVER/ks/$pkg_name");
	    if ($? != 0) {
		showerror("Conversion from Workstation to Server failed");
		return($EXIT_RHWS_CONVERT);
	    }
	}

	my $tmp_logfile = "/tmp/RHNupdate.log";
	my $hostname = get_hostname();
	my $recipient = "managedservicesar\@teleflora.com";

	# Do our OS Upgrade here.
	# Note, no upgrades for RH72, FC3, or FC5.

	my $exit_status = $EXIT_OK;

	if ($OS eq "RHEL4") {

	    purge_rpms();

	    # "-uf" = update system and force installation
	    my $up2date_cmd = '/usr/sbin/up2date -uf';

	    system("$up2date_cmd | tee $tmp_logfile");
	    if ($? == 0) {

		# notification of results by email
		system("mail -s \"RHN update for $hostname\" $recipient < $tmp_logfile");

		# Only install the digi drivers into the kernel on rti systems.
		if ($RTI) {
		    if (update_ospatches_kernel_fixup() != 0) {
			$exit_status = $EXIT_DIGI_DRIVERS;
		    }
		}
	    }

	    else {
		showerror("Up2date update command returned non-zero status");
		$exit_status = $EXIT_UP2DATE;
	    }
	}

	elsif ( ($OS eq "RHEL6") || ($OS eq "RHEL5") || ($OS eq "RHWS5") ) {

	    # comment out the line in /etc/yum.conf which disables kernel updates
	    update_yum_config();

	    purge_rpms();

	    # run the yum clean command - send output in real time to stdout
	    # as well as saving output in both log files.
	    open(LFH, '>>', $LOGFILE_PATH);
	    open(TLFH, '>>', $tmp_logfile);
	    my $cmd = "yum clean -y all";
	    loginfo($cmd);
	    if (open(my $pfh, '-|', "$cmd 2>&1")) {
		while (<$pfh>) {
		    print $_;
		    print LFH $_;
		    print TLFH $_;
		}
		if (!close($pfh)) {
		    showerror("command ($cmd) returned non-zero status: $?");
		    $exit_status = $EXIT_YUM_UPDATE;
		}
	    }
	    else {
		showerror("can't open command as pipe: $cmd");
		$exit_status = $EXIT_YUM_UPDATE;
	    }
	    close(LFH);
	    close(TLFH);

	    if ($exit_status == $EXIT_OK) {

		# run the yum update command - send output in real time to stdout
		# as well as saving output in both log files.
		open(LFH, '>>', "$LOGFILE_PATH");
		open(TLFH, '>>', "$tmp_logfile");
		my $cmd = "yum update -y";
		loginfo("$cmd");
		if (open(my $pfh, '-|', "$cmd 2>&1")) {
		    while (<$pfh>) {
			print $_;
			print LFH $_;
			print TLFH $_;
		    }
		    if (close($pfh)) {
			# send both results via email
			system("mail -s \"RHN update for $hostname\" $recipient < $tmp_logfile");

			# only install the digi drivers into the kernel on non-RHEL6 RTI systems.
			if ($RTI && ($OS ne "RHEL6") ) {
			    if (update_ospatches_kernel_fixup() != 0) {
				$exit_status = $EXIT_DIGI_DRIVERS;
			    }
			}

			# only fixup the "initscripts" on RHEL6 Daisy system
			if ($DAISY && ($OS eq "RHEL6")) {
			    if (update_ospatches_initscripts_fixup() != 0) {
				$exit_status = $EXIT_INITSCRIPTS;
			    }
			}
		    }
		    else {
			showerror("command ($cmd) returned non-zero status: $?");
			$exit_status = $EXIT_YUM_UPDATE;
		    }
		}
		else {
		    showerror("can't open command as pipe: $cmd");
		    $exit_status = $EXIT_YUM_UPDATE;
		}
		close(LFH);
		close(TLFH);
	    }
	}

	if (-f $tmp_logfile) {
	    unlink($tmp_logfile);
	}

	if ($exit_status == 0) {

	    # It is possible that our OS updated sshd, in which case,
	    # the 'tfremote' hard links should be re-established. By doing so,
	    # tfremote will use the 'newly updated' sshd.
	    if (-f "/usr/sbin/tfremote" && -f "/usr/sbin/sshd") {
		    system("rm -f /usr/sbin/tfremote");
		    system("ln /usr/sbin/sshd /usr/sbin/tfremote");
	    }

	    # reset intrusion detection database
	    reset_aide_database();
	}


        $timestamp = strftime("%Y%m%d%H%M%S", localtime());
	showinfo("Timestamp: $timestamp");

	showinfo("End Installing OS Patches...");

	return($exit_status);
}


#
# If the "initscripts" rpm package has been updated, two unwanted
# files will have been copied to "/etc/init".
#
# Returns 0 on success or 1 on error
#
sub update_ospatches_initscripts_fixup
{
    my @unwanted_files = qw(
	/etc/init/start-ttys.conf
	/etc/init/tty.conf
    );

    foreach (@unwanted_files) {
	unlink($_);
	return(1) if (-f $_);
    }

    return(0);
}


#
# For each installed kernel, install the Digi drivers if not already installed.
# If the drivers are needed and not downloaded yet, download them.
#
# Returns 0 on success or 1 on error
#
sub update_ospatches_kernel_fixup
{
	my $tmpdir = '/tmp';			# use /tmp for working storage
	my $kern_modules_path = '/lib/modules';	# location of kernel modules
	my $kern_drivers_dir = 'misc';		# subdir for kernel drivers

	# the digi drivers tar file is assumed to contain two files: dgap.ko and dgrp.ko
	my $digi_drivers_tarfile = 'ES4Digidrivers.tar.gz';
	if ( ($OS eq "RHEL5") || ($OS eq "RHWS5") ) {
		$digi_drivers_tarfile = 'WS5digi.tar.gz';
	}
	my $digi_dgap_file = 'dgap.ko';
	my $digi_dgrp_file = 'dgrp.ko';
	my $digi_drivers_path = "$tmpdir/$digi_drivers_tarfile";
	my $digi_dgap_path = "$tmpdir/$digi_dgap_file";
	my $digi_dgrp_path = "$tmpdir/$digi_dgrp_file";

	# the digi drivers tar file should be available on the ostools web site
	my $ostools_url = "http://$TFSERVER/ostools";
	my $digi_drivers_url = "$ostools_url/$digi_drivers_tarfile";

	showinfo("Installing Digi Drivers...");

	my @kernel_list = glob("$kern_modules_path/*");
	unless (@kernel_list) {
	    showerror("kernel modules directory empty: $kern_modules_path");
	    return(1);
	}

	foreach my $thiskern (@kernel_list) {

	    showinfo("Verifying presence of Digi drivers for kernel: $thiskern");

	    #
	    # This path should look like: /lib/modules/2.6.18-194.8.1.el/misc
	    #
	    my $kern_drivers_path = "$thiskern/$kern_drivers_dir";

	    # if there is no drivers dir, make one
	    unless (-d "$kern_drivers_path") {
		system("mkdir $kern_drivers_path");
	    }
	    unless (-d "$kern_drivers_path") {
		showerror("Can't make kernel drivers directory: $thiskern");
		return(1);
	    }

	    # if both drivers are present, nothing to do
	    if (-f "$kern_drivers_path/$digi_dgap_file" &&
		-f "$kern_drivers_path/$digi_dgrp_file") {
		showinfo("Digi drivers already installed for kernel: $thiskern");
		next;
	    }

	    #
	    # One or more drivers not installed...
	    # So download drivers if not already downloaded and then install them
	    #
	    unless (-f $digi_drivers_path) {

		showinfo("Downloading Digi Drivers from $digi_drivers_url...");

		system("curl -s -o $digi_drivers_path $digi_drivers_url");
		system("tar ztf $digi_drivers_path > /dev/null 2> /dev/null");
		if ($? != 0) {
		    showerror("download of digi drivers failed: $digi_drivers_url");
		    return(1);
		}
		system("cd $tmpdir && tar zxf $digi_drivers_path");
		if ($? != 0) {
		    showerror("untar of digi drivers failed: $digi_drivers_path.");
		    return(1);
		}
	    }

	    showinfo("Installing Digi Drivers for kernel: $thiskern");

	    system("cp $digi_dgap_path $kern_drivers_path");
	    system("chown root:root $kern_drivers_path/$digi_dgap_file");
	    system("chmod 664 $kern_drivers_path/$digi_dgap_file");
	    system("cp $digi_dgrp_path $kern_drivers_path");
	    system("chown root:root $kern_drivers_path/$digi_dgrp_file");
	    system("chmod 664 $kern_drivers_path/$digi_dgrp_file");
	}

	if (-f $digi_drivers_path) {
	    system("rm $digi_drivers_path");
	    system("rm $digi_dgap_path");
	    system("rm $digi_dgrp_path");
	}

	return(0);
}

sub update_yum_config
{
        my $timestamp = strftime("%Y-%m-%d_%H%M%S", localtime());
        my $conf_file = '/etc/yum.conf';
        my $new_conf_file = "$conf_file-$timestamp";

        # grep returns non-zero if search pattern not found
        system("grep '^exclude=kernel' $conf_file > /dev/null 2> /dev/null");
        if ( (WIFEXITED($?) != 0) && (WEXITSTATUS($?) != 0) ) {
                return(1);
        }

        unless (-s "$conf_file") {
                showerror("$conf_file does not exist...  Will skip edits.");
                return(2);
        }

        unless (open(OLD, "< $conf_file")) {
                showerror("Could not open $conf_file for read... Will skip edits.");
                return(3);
        }

        unless (open(NEW, "> $new_conf_file")) {
                showerror("Could not open $new_conf_file for write... Will skip edits.");
                close(OLD);
                return(4);
        }
        while (<OLD>) {
                chomp;

                if ( /^(\s*)exclude=kernel/ ) {
                        print(NEW "# --- following line commented out ---\n");
                        print(NEW "# $_\n");
			next;
                }

                # By default, just copy what we read.
                print(NEW "$_\n");

        }

        close(OLD);
        close(NEW);

        # If we created a new conf file that is zero sized, that is bad.
        if (-z $new_conf_file) {
                showerror("Copy of $conf_file is a zero size file. Will skip edits.");
                system("rm $new_conf_file");
                return(5);
        }

        # Assume conf file was successfully transformed... so replace the old one with the new.
        system("chmod --reference=$conf_file $new_conf_file");
        system("chown --reference=$conf_file $new_conf_file");
        system("mv $new_conf_file $conf_file");

	return(0);
}


# Just grab and run an install script from our website.
# Let that script do the hard work.
sub update_ostools
{
	print("Updating Teleflora OS Tools...\n");
	system("wget -O - http://$TFSERVER/ostools/install-ostools-1.14.pl | sudo perl - --update");
	return($?);
}


# Disable X-WIndows for those older FC3, FC5 and RHEL4 customers.
# This falls under the PCI auspice of "disabling sfotware without a business need."
sub modify_inittab
{
	# inittab
	if (-f "/etc/inittab") {
		showinfo("Modifying inittab.");
		open(OLDFILE, "< /etc/inittab");
		open(NEWFILE, "> /etc/inittab.$$");
		while(<OLDFILE>) {

			# PCI 2.2.2
			if(/(:)(\s*)(initdefault)(\s*)(:)/i) {
				print(NEWFILE "id:3:initdefault:\n");
				next;
			}

			print(NEWFILE);
		}
		close(OLDFILE);
		close(NEWFILE);

		if(! -z "/etc/inittab.$$") {
			system("mv /etc/inittab.$$ /etc/inittab");
		}
	}

	return(0);
}

#
# Make sure the daily 'locate' db update is enabled.  And to do that,
# we need to make sure the contents of /etc/updatedb.conf file is appropriate.
#
# The updatedb job is run out of /etc/cron.daily cron job so it will
# normally be run every day.  However, we believe that FC3 installations
# of Linux have some logic like the following in the "locate" cron job:
#
#	. /etc/updatedb.conf
#	[ "$DAILY_UPDATE" != "YES" ] && exit 0
#
# So, if a line like "DAILY_UPDATE=YES" does not appear in the
# /etc/updatedb.conf file, then the cron job will not run.
#
# We do know that RHEL5 does not have this logic and in fact updatedb
# outputs a diagnostic if the conf file is changed to add "DAILY_UPDATE"...
#
sub modify_updatedb
{
    #
    # We were using the perl "constant" module for the constants below
    # but RH72 doesn't have it so we must resort to just local vars.
    #
    my $UPDATEDB_REMOVE_IT = 0;
    my $UPDATEDB_ADD_IT = 1;
    my $UPDATEDB_LEAVE_IT = 2;

    my $conf_file = "$ALTROOT/etc/updatedb.conf";

    unless (-f $conf_file) {
	showinfo("$conf_file: file does not exist");
	return(0);
    }

    my $edit_action = $UPDATEDB_REMOVE_IT;

    #
    # Set flag for "remove it" systems.
    #
    if ($OS eq "RHEL6" || $OS eq "RHEL5" || $OS eq "RHEL4" || $OS eq "FC5" || $OS eq "RH72") {
	$edit_action = $UPDATEDB_REMOVE_IT;

    #
    # Set flag for "add it" systems.
    #
    } elsif ($OS eq "FC3") {
	$edit_action = $UPDATEDB_ADD_IT;

    #
    # anything else, we don't know what to do, so do nothing.
    #
    } else {
	showinfo("No action taken for OS type: $OS");
	return(0);
    }

    #
    # OK, we have to modify the file.
    #
    showinfo("Modifying updatedb config file: $conf_file.");
    open(OLDFILE, "< $conf_file");
    open(NEWFILE, "> $conf_file.$$");
    while (<OLDFILE>) {
	if (/^(\s*)(DAILY_UPDATE)(.+)(YES)/) {
	    if ($edit_action == $UPDATEDB_REMOVE_IT) {
		print(NEWFILE "### $PROGNAME $TIMESTAMP: following line commented out\n");
		print(NEWFILE "### $_");
		next;
	    } elsif ($edit_action == $UPDATEDB_ADD_IT) {
		$edit_action = $UPDATEDB_LEAVE_IT;
	    }
	}
	print(NEWFILE $_);
    }

    if ($edit_action == $UPDATEDB_ADD_IT) {
	print(NEWFILE "DAILY_UPDATE=YES\n");
    }
    close(OLDFILE);
    close(NEWFILE);

    if (! -z "$conf_file.$$") {
	system("mv $conf_file.$$ $conf_file");
	if ($? != 0) {
	    showerror("Can't move new instance of $conf_file into place");
	    showinfo("$conf_file unchanged");
	    system("rm $conf_file.$$");
	    return(1);
	}
    }

    showinfo("New instance written: $conf_file");

    return(0);
}

#
# Clear out the login banner file, /etc/motd.
#
# There is no PCI requirement driving this file's contents.
#
sub modify_motd
{
        my $conf_file = "$ALTROOT/etc/motd";
        my $new_conf_file = "$conf_file.$$";

	showinfo("Truncating Login Banner File: $conf_file...");

        if (! -f "$conf_file") {
                showerror("File does not exist: $conf_file...  Will skip edits.");
                return(1);
        }

	# make new empty motd file
	system("touch $new_conf_file");

	# did we successfully make a new motd file?
        if (! -f "$new_conf_file") {
                showerror("Can't make new motd file: $new_conf_file...  Will skip edits.");
                return(1);
        }

        # replace the old one with the new.
        system("chmod --reference=$conf_file $new_conf_file");
        system("chown --reference=$conf_file $new_conf_file");
        system("mv $new_conf_file $conf_file");
        if ($? != 0) {
                showerror("Can't replace old motd file.");
		system("rm -f $new_conf_file");
                return(1);
        }

	return(0);
}


#
# Set the "passdb backend"  parameter to a value of "smbpasswd"
# in the samba config file.  This is a "global" section parameter.
# This must be done for RHEL6 systems to be backwards compatabile
# with the way the pre-RHEL6 systems were configured.
#
sub samba_set_passdb
{
    my $parameter = "passdb backend = smbpasswd";
    my $parameter2 = "smb passwd file = /etc/samba/smbpasswd";

    my $conf_file = "$ALTROOT/etc/samba/smb.conf";
    my $new_conf_file = "$conf_file.$$";

    unless (-f $conf_file) {
	showerror("Samba config file does not exist: $conf_file");
	return($EXIT_SAMBA_CONF);
    }

    # Do nothing if a modified conf file is already in place.
    system("grep '$parameter' $conf_file > /dev/null 2> /dev/null");
    if ($? == 0) {
	showinfo("Samba config file already appears to be modified: $conf_file");
	return(0);
    }

    unless (open(OLD, '<', $conf_file)) {
	showerror("Can't open existing Samba config file: $conf_file");
	return($EXIT_SAMBA_CONF);
    }
    unless (open(NEW, '>', $new_conf_file)) {
	showerror("Can't make new Samba config file: $new_conf_file");
	return($EXIT_SAMBA_CONF);
    }

    #
    # Copy all lines from old to new, but after the global section dec,
    # write the new parameter into the new conf file.
    #
    while (<OLD>) {
	if (/^\s*\[global\]/) {
	    print(NEW $_);
	    print(NEW "#Following lines added by $PROGNAME, $CVS_REVISION, $TIMESTAMP\n");
	    print(NEW "$parameter\n");
	    print(NEW "$parameter2\n");
	    next;
	}
	else {
	    print(NEW);
	}
    }

    close(OLD);
    close(NEW);

    # If we created a new conf file that is zero sized, that is bad.
    if (-z $new_conf_file) {
	showerror("The copy of the Samba $conf_file is a zero size file");
	showerror("Samba config file NOT modified with new paramter: $parameter");
	system("rm $new_conf_file");
	return($EXIT_SAMBA_CONF);
    }

    # Assume conf file was successfully transformed...
    # so replace the old one with the new.
    system("chmod --reference=$conf_file $new_conf_file");
    system("chown --reference=$conf_file $new_conf_file");
    system("cp $new_conf_file $conf_file");

    system("/sbin/service smb restart");

    loginfo("Samba config file modified with new paramter: $parameter");

    return($EXIT_OK);
}


#
# Make the UIDs in the "smbpasswd" file match those in /etc/passwd.
#
sub samba_rebuild_passdb
{
    my $conf_file = "$ALTROOT/etc/samba/smbpasswd";
    if ($OS eq "RHEL6") {
	unless (-e $conf_file) {
	    $conf_file = "$ALTROOT/var/lib/samba/private/smbpasswd";
	}
    }
    my $new_conf_file = "$conf_file.$$";

    unless (-f $conf_file) {
	showerror("Samba config file does not exist: $conf_file");
	return($EXIT_SAMBA_CONF);
    }

    unless (open(OLD, '<', $conf_file)) {
	showerror("Can't open existing Samba config file: $conf_file");
	return($EXIT_SAMBA_CONF);
    }
    unless (open(NEW, '>', $new_conf_file)) {
	showerror("Can't make new Samba config file: $new_conf_file");
	return($EXIT_SAMBA_CONF);
    }

    #
    # Copy all lines from old to new, but adjust the UID field if necessary.
    #
    # The structure of an smbpasswd entry is:
    #
    # username:uid:lanman_hash:nt_hash:flags:pw_lct
    #
    # username    - the user's login name.
    # uid         - the user's UID
    # lanman_hash - Windows stuff
    # nt_hash     - Windows stuff
    # flags       - Various single-character flags representing the type and
    #               state of the user's account.
    # pw_lct      - the timestamp of the user's last successful password change
    #
    my $file_modified = 0;
    while (<OLD>) {
	my $line = $_;
	if ($line =~ /^(\S+):(\d+):(.*)$/) {
	    my $username = $1;
	    my $uid = $2;
	    my $remainder = $3;

	    my $system_uid = getpwnam($username);
	    if (defined($system_uid)) {
		if ($uid ne $system_uid) {
		    $line = "$username" . ":" . "$system_uid" . ":" . "$remainder" . "\n";
		    $file_modified = 1;
		}
	    }
	}
	print(NEW $line);
    }

    close(OLD);
    close(NEW);

    if ($file_modified) {
	# If we created a new conf file that is zero sized, that is bad.
	if (-z $new_conf_file) {
	    showerror("The copy of the Samba $conf_file is a zero size file");
	    showerror("Samba $conf_file NOT modified with new UID fields");
	    system("rm $new_conf_file");
	    return($EXIT_SAMBA_CONF);
	}

	# Assume conf file was successfully transformed...
	# so replace the old one with the new.
	system("chmod --reference=$conf_file $new_conf_file");
	system("chown --reference=$conf_file $new_conf_file");
	system("cp $new_conf_file $conf_file");

	loginfo("Samba $conf_file modified with new UID fields");
    }
    else {
	system("rm $new_conf_file");
	loginfo("Samba $conf_file did not need any modification with new UID fields");
    }

    return($EXIT_OK);
}



sub modify_samba
{
	my ($pos_type) = @_;

	unless ($pos_type eq $POS_TYPE_RTI || $pos_type eq $POS_TYPE_DAISY) {
		showerror("POS type not specified - can't generate appropriate Samba conf file");
		return($EXIT_SAMBA_CONF);
	}

	my $conf_file = "$ALTROOT/etc/samba/smb.conf";

	unless (-f $conf_file) {
		showerror("Samba config file does not exist: $conf_file");
		return($EXIT_SAMBA_CONF);
	}


	####################################
	#### RTI Samba Config
	####################################
	if ($pos_type eq $POS_TYPE_RTI) {

		# Do nothing if a modified conf file is already in place.
		system("grep usr2 $conf_file > /dev/null 2> /dev/null");
		if ($? == 0) {
			showinfo("Samba config file already appears to be modified: $conf_file");
			return($EXIT_SAMBA_CONF);
		}

		unless (open(FILE, '>', $conf_file)) {
			showerror("Can't open Samba config file: $conf_file");
			return($EXIT_SAMBA_CONF);
		}

		loginfo("Generating new RTI Samba config file");

		print(FILE "# RTI Samba Configuration\n");
		print(FILE "# Automatically generated by $PROGNAME $CVS_REVISION $TIMESTAMP\n");
		print(FILE "#\n");
		print(FILE << 'xxxEOFxxx');


#
#======================= Global Settings =====================================
[global]


server string = RTI Server
hosts allow = 192.168. 127. 10. 172.16.
domain master = no
local master = no
preferred master = no
os level = 0
dns proxy = no
smb ports = 139
log file = /var/log/samba/%m.log
max log size = 50
socket options = TCP_NODELAY SO_RCVBUF=8192 SO_SNDBUF=8192
idmap uid = 16777216-33554431
idmap gid = 16777216-33554431
username map = /etc/samba/smbusers
passdb backend = smbpasswd
smb passwd file = /etc/samba/smbpasswd
security = user
server signing = auto



#======================= Printer Settings =====================================
printcap name = /etc/printcap
cups options = raw

[printers]
	comment = All Printers
	path = /var/spool/samba
	browseable = no
	printable = yes

#======================= Shares =====================================
[odbc_dict]
	comment = ODBC Data Dictionary
	path = /usr2/bbx/odbc_dict
	writeable = yes
	guest ok = yes
[mapping]
        comment = Mapping
        path = /usr2/bbx/delivery
        writeable = yes
        guest ok = yes
	create mask = 0666
[tmp]
        comment = Temporary Files
        path = /tmp
        writeable =  yes
   	read only = no
   	public = yes
        guest ok = yes
[wirerec]
        comment = Wire Service Reconciliations
        path = /usr2/bbx/bbxtmp
        read only = no
        guest ok = no

[reports]
        comment = RTI Reports
        path = /usr2/bbx/reports
        read only = no
        guest ok = no

xxxEOFxxx
	}


	####################################
	#### Daisy Samba Config
	####################################
	elsif ($pos_type eq $POS_TYPE_DAISY) {

		# Do nothing if a modified conf file is already in place.
		system("grep daisy $conf_file > /dev/null 2> /dev/null");
		if ($? == 0) {
			showinfo("Samba config file already appears to be modified: $conf_file");
			return($EXIT_SAMBA_CONF);
		}

		unless (open(FILE, '>', $conf_file)) {
			showerror("Can't open Samba config file: $conf_file");
			return($EXIT_SAMBA_CONF);
		}

		loginfo("Generating new Daisy Samba config file.");

		print(FILE "# Daisy Samba Configuration\n");
		print(FILE "# Automatically generated by $PROGNAME $CVS_REVISION $TIMESTAMP\n");
		print(FILE "#\n");
		print(FILE << 'xxxEOFxxx');


#
#======================= Global Settings =====================================
[global]

workgroup = daisy
server string = daisyhost
netbios name = daisyhost
hosts allow = 192.168. 127. 10. 172.16.
writable = yes
domain master = no
local master = no
preferred master = no
os level = 0
dns proxy = no
smb ports = 139
log file = /var/log/samba/%m.log
max log size = 50
socket options = TCP_NODELAY SO_RCVBUF=8192 SO_SNDBUF=8192
idmap uid = 16777216-33554431
idmap gid = 16777216-33554431
username map = /etc/samba/smbusers
guest account = daisy
encrypt passwords = yes
passdb backend = smbpasswd
smb passwd file = /etc/samba/smbpasswd
security = user
server signing = auto


#======================= Printer Settings =====================================
printcap name = /etc/printcap
cups options = raw

[printers]
	comment = All Printers
	path = /var/spool/samba
	browseable = yes
	guest ok = yes
	writeable = no
	printable = yes
	public = yes
	

#======================= Shares =====================================

[homes]
	comment = Home Directories
	browseable = no
        writable = no

[export]
        comment = Daisy share
        path = /d/daisy/export
        writable = yes
	browseable = yes
	guest ok = yes
	printable = no
	public = yes
xxxEOFxxx
	}

	close(FILE);

	system("/sbin/service smb restart");
}



#
# Remove older kernels except three most recent.
# This to work around undersized /boot partitions on our older
# RHEL4 and RHWS5 machines.
#
sub purge_rpms
{
	my $thiskernel = "";
	my $thisrpm = "";
	my $thisfile = "";
	my $keepcount = 0;
	my $bootdir = '/boot';
	my $kernels = "$bootdir/vmlinuz-*";

	showinfo("Removing Old Kernels...");

	# Keep the "n" most recent kernels, where "n" == $KEEPKERNELS.
	# Remove everything else.

	showinfo("Searching $bootdir for Old Kernels...");

	open(FILES, "ls -dt $kernels 2>/dev/null |");
	while (<FILES>) {
		chomp;
		$thiskernel = $_;

		#
		# The kernel file must exist and should since we just found it
		#
		next unless (-e $thiskernel);

		#
		# Keep last $KEEPKERNELS kernels
		#
		$keepcount++;
		next if($keepcount <= $KEEPKERNELS);

		open(RPM, "rpm -qf $thiskernel |");
		while(<RPM>) {
			chomp;
			$thisrpm = $_;
			showinfo("Removing Kernel  $thisrpm...");
			system("rpm -e $thisrpm");
		}
		close(RPM);
	}
	close(FILES);

	# Any other RPM packages which we should delete here?
	# eg. 'X', GCC, etc.


	return(0);
}


sub java_package_name
{
    my ($java_version) = @_;

    my $jre_arch = "";
    my $jre_pkg = "";

    if ($ARCH eq "i386") {
	$jre_arch = "i586";
    }
    elsif ($ARCH eq "x86_64") {
	$jre_arch = "x64";
    }
    else {
	showerror("Can't happen: unsupported ARCH: $ARCH");
    }

    if ($jre_arch) {
	$jre_pkg = "jre-" . $java_version . "-linux-" . $jre_arch . "-rpm.bin";
    }

    return($jre_pkg);
}


sub java_download
{
    my ($jre_dir, $jre_pkg) = @_;

    my $retval = 0;

    my $jre_path = "$jre_dir/$jre_pkg";

    unless (-f $jre_path) {
	# Curl options:
	# -f : set exit value on failure
	# -s : no progress indication
	my $curl_fail = "-f";
	my $curl_silent = ($VERBOSE) ? "" : "-s";
	my $curl_opts = "$curl_fail $curl_silent";
	my $curl_cmd = "curl $curl_opts -o $jre_path http://$TFSERVER/ks/$jre_pkg";
	if ($DRY_RUN) {
	    system("echo $curl_cmd");
	}
	else {
	    system("$curl_cmd");
	    if ($? != 0) {
		showerror("Could not download Java package: $jre_pkg");
		$retval = $EXIT_JAVA_DOWNLOAD;
	    }
	}
    }

    return($retval);
}


sub java_install
{
    my ($jre_dir, $jre_pkg) = @_;

    my $retval = 0;

    my $jre_path = "$jre_dir/$jre_pkg";

    if ($DRY_RUN) {
	system("echo chmod a+rx $jre_path");
	system("echo $jre_path");
    }
    else {
	if (-f $jre_path) {
	    system("rm -rf /tmp/updateos.$$");
	    system("mkdir /tmp/updateos.$$");
	    system("chmod a+rx $jre_path");
	    system("cd /tmp/updateos.$$ && $jre_path");
	    if ($? != 0) {
		showerror("Could not install Java package: $jre_pkg");
		$retval = $EXIT_JAVA_INSTALL;
	    }
	    else {
		system("rm -rf /tmp/updateos.$$");
	    }
	}
	else {
	    showerror("Java installation package does not exist: $jre_pkg");
	    $retval = $EXIT_JAVA_INSTALL;
	}
    }

    return($retval);
}


sub install_java
{
    my ($java_version) = @_;

    my $retval = 0;
    my $jre_dir = "/tmp";

    my $jre_pkg = java_package_name($java_version);
    if ($jre_pkg eq "") {
	$retval = $EXIT_JAVA_VERSION;
    }
    else {
	showinfo("Downloading Java installation package...");
	$retval = java_download($jre_dir, $jre_pkg);
	if ($retval == 0) {
	    showinfo("Installing Java...");
	    $retval = java_install($jre_dir, $jre_pkg);
	}
    }

    if ($retval == $EXIT_OK) {
	showinfo("Java package installed: $jre_pkg");
    }

    return($retval);
}


#
# Install OS pre-requisites for RTI v14.
#
sub install_rtiv14_packages
{
	my $if_name = $_[0];
	my @array = ();
	my $bbj_nozip = "";
	my $line = "";
	my $need_to_change = 0;
	my $xmx = "512m";

	my $rhn_system_id = get_rhn_system_id();
	if ($rhn_system_id eq "") {
	    showerror("system not registered with Red Hat Network");
	    return($EXIT_RHN_NOT_REGISTERED);
	}

	showinfo("Installing RTI v14 Software Dependencies...");
	showinfo("Using Teleflora Package Server \"$TFSERVER\"");

	# if booting "DHCP", log an error and exit
	my $conf_file = "$ALTROOT/etc/sysconfig/network-scripts/ifcfg-$if_name";
	my $pattern = ($OS eq "RHEL6") ? 'BOOTPROTO=\"dhcp\"' : 'BOOTPROTO=dhcp';
	system("grep $pattern $conf_file > /dev/null 2> /dev/null");
	if ($? == 0) {
		showerror("Network must be configured with static ip first");
		return($EXIT_RTI14);
	}

	remount_filesystem("/teleflora", "/usr2");

	if ($OS eq "RHEL5" || $OS eq "RHEL6") {
	    # Install Fetchmail
	    system("yum -y install fetchmail");

	    # RHEL6 no longer has uucp - end of an era
	    if ($OS ne "RHEL6") {
		    system("yum install -y uucp");
	    }

	    # some RTI code ksh dependent
	    system("yum install -y ksh");
	}

	create_default_users("rti");

	#system("/usr2/bbx/bin/rtiuser.pl --add tfsupport");
	#system("/usr2/bbx/bin/rtiuser.pl --enable-admin tfsupport");
	#system("/usr2/bbx/bin/rtiuser.pl --add rti");

	# Install apache
	if( ($OS eq "RHEL6") || ($OS eq "RHEL5") || ($OS eq "RHWS5") ) {
	    showinfo("Installing Apache...");
	    system("yum -y install httpd");
	}
	if("$OS" eq "RHEL4") {
	    showinfo("Installing Apache...");
	    system("up2date --install httpd");
	}

	# Samba
	system("grep delivery /etc/samba/smb.conf > /dev/null 2> /dev/null");
	if($? != 0) {
		modify_samba($POS_TYPE_RTI);
	}

	create_default_samba_users("rti");

	# Install Java

	showinfo("Installing Java JRE...");
	if (install_java($JAVA_VERSION) != 0) {
	    return($EXIT_JAVA_INSTALL);
	}
	
	# Install BBj
	if(-d "/usr2/basis") {
		loginfo("Moving current /usr2/basis/ to /usr2/basis-backup/...");
		system("mv /usr2/basis /usr2/basis-backup");
	} else {
		loginfo("/usr2/basis/ did not previously exist...");
	}

	showinfo("Installing BBj...");
	if(! -f "/tmp/$BBJ") {
		system("wget --proxy=off --cache=off -O /tmp/$BBJ http://$TFSERVER/ks/$BBJ");
		if(! -f "/tmp/$BBJ") {
			showerror("Could not download file http://$TFSERVER/ks/$BBJ");
		}
	}

	#Create bbjinstallsettings.txt
	open(BBJINST, "> /tmp/bbjinstallsettings.txt");
        print(BBJINST << 'EOF');
-V LICENSE_ACCEPT_BUTTON="true"
-V LICENSE_REJECT_BUTTON="false"
-P installLocation="/usr2/basis/"
-V IS_SELECTED_INSTALLATION_TYPE=custom
-P BBjFtrBean.active=true
-P BBjThinClientFtr_bean.active=true
-P BBjSvcs_FtrBean.active=true
-P BBjEUTFtr_bean.active=true
-P EMFeatureBean.active=true
-P adminFeatureBean.active=true
-P BBjUtilsFtr_bean.active=true
-P GMLFeatureBean.active=true
-P BWUFeatureBean.active=true
-P STDFeatureBean.active=true
-P EXTFeatureBean.active=true
-P MKRecoverFeatureBean.active=true
-P AutoLicFeatureBean.active=true
-P CLlibFeatureBean.active=true
-P JLlibFeatureBean.active=true
-P BBjDevelToolsFtr_bean.active=true
-P perfAnalyzerFeatureBean.active=false
-P compilerListerFeatureBean.active=true
-P configuratorFeatureBean.active=false
-P guiBuilderFeatureBean.active=false
-P blmFeatureBean.active=true
-P BLMadminFeatureBn.active=true
-P basisIDEfeatureBean.active=false
-P documentationFeatureBean.active=false
-P jdbcFeatureBean.active=true
-P TrainingfeatureBean.active=true
-P DemosBean.active=true
-P BaristafeatureBean.active=false
-P AddOnfeatureBean.active=false
EOF
	close(BBJINST);
	if ( (-f "/tmp/$BBJ") && (-f "/tmp/bbjinstallsettings.txt")) {
		$bbj_nozip = $BBJ;
		$bbj_nozip =~ s/\.gz//g; # blah.jar.gz -> blah.jar
		system("gzip -dc /tmp/$BBJ > /tmp/$bbj_nozip");
		system("java -jar /tmp/$bbj_nozip -options /tmp/bbjinstallsettings.txt -silent");
	}


	# Figure out how much memory we have, and hence, what our default java "xmx" values should be set to.
	open(MEM, "< /proc/meminfo");
	while(<MEM>) {
		chomp;
		if(/MemTotal:/) {
			@array = split(/\s+/);

			# Less than a gig of memory
			if( int($array[1]) < 1000000) {
				$xmx = "512m";

			# Betweeen 1GB and 2GB
			} elsif( (int($array[1]) > 1000000) && (int($array[1]) < 2000000) )  {
				$xmx = "768m";

			# Betweeen 2GB and 3GB
			} elsif( ( int($array[1]) > 2000000) && ( int($array[1]) < 3000000))  {
				$xmx = "1024m";

			# Betweeen 3GB and 4GB
			} elsif( ( int($array[1]) > 3000000) && ( int($array[1]) < 4000000))  {
				$xmx = "1536m";

			# 4GB ++
			} else {
				$xmx = "2048m"
			}
		}
	}
	close(MEM);


	# Update BBj.properties File
	if(-f "/usr2/basis/cfg/BBj.properties") {
		loginfo("Backing up BBj.properties");
		system("mv /usr2/basis/cfg/BBj.properties /usr2/basis/cfg/BBj.properties.original");
	}
	loginfo("Updating BBj.properties");
	open(BBJPROP, "> /usr2/basis/cfg/BBj.properties");
	print(BBJPROP "# BBj Properties File\n");
	print(BBJPROP "# This file auto-generated by $PROGNAME $CVS_REVISION $TIMESTAMP\n");
	print(BBJPROP "# This file auto-generated on date: " . localtime() . "\n");
	print(BBJPROP "#\n");
	print(BBJPROP "#\n");


	print(BBJPROP << 'EOF');
#BBj Services Properties
basis.cacheDirectory=/usr2/basis/cache
basis.java.args.BBjCpl=-XX\:NewSize\=24m -client
basis.java.args.BBjLst=-XX\:NewSize\=24m -client
EOF
print(BBJPROP "basis.java.args.BBjServices=-Xmx$xmx -Xms256m ");

	print(BBJPROP << 'EOF');
 -XX\:MaxPermSize\=96m -XX\:NewSize\=64m -XX\:+UseConcMarkSweepGC -XX\:+CMSIncrementalMode -XX\:CMSIncrementalSafetyFactor\=20 -XX\:CMSInitiatingOccupancyFraction\=70 -XX\:MaxTenuringThreshold\=4 -XX\:SurvivorRatio\=14 -server -XX\:CompileCommandFile\=/usr2/basis/cfg/.hotspot_compiler -Dnetworkaddress.cache.ttl\=10 -Dsun.net.inetaddr.ttl\=10 -Djava.awt.headless\=true -verbose\:gc -Xloggc\:/usr2/basis/log/gc.log -XX\:+PrintGCDetails -XX\:+PrintGCTimeStamps -XX\:+PrintGCApplicationConcurrentTime -XX\:+PrintGCApplicationStoppedTime
basis.java.args.BasisIDE=-XX\:CompileCommandFile\=/usr2/basis//cfg/.hotspot_compiler
basis.java.args.Default=-Xmx128m -Xms128m -XX\:NewRatio\=4 -XX\:MaxPermSize\=128m -XX\:NewSize\=24m -client -XX\:CompileCommandFile\=/usr2/basis//cfg/.hotspot_compiler
basis.java.classpath=/usr2/basis//lib/BBjUtil.jar\:/usr2/basis//lib/BBjIndex.jar\:/usr2/basis//lib/FontChooser.jar\:/usr2/basis//lib/DemoClientFiles.jar\:/usr2/basis//lib/ClientObjects.jar\:/usr2/basis//lib/examples.jar\:/usr2/basis//lib/jfreechart-experimental.jar\:/usr2/basis//lib/jna.jar\:/usr2/basis//lib/swingx.jar\:/usr2/basis//lib/TimingFramework.jar\:/usr/java/latest/lib/ext/jdic_stub_unix.jar
basis.java.jvm.BBjServices=/usr/java/latest/bin/java
basis.java.jvm.Default=/usr/java/latest/bin/java
basis.java.skin=
basis.pdf.fontpath=
com.basis.bbj.bridge.BridgeServer.bindAddr=0.0.0.0\:2007
com.basis.bbj.bridge.BridgeServer.maxClients=100
com.basis.bbj.bridge.BridgeServer.start=false
com.basis.bbj.comm.FacadeFactory.suppressClientObjErrorsOnVoidMethods=true
com.basis.bbj.comm.InterpreterServer.backLog=110
com.basis.bbj.comm.InterpreterServer.bindAddr=127.0.0.1\:2005
com.basis.bbj.comm.InterpreterServer.maxClients=500
com.basis.bbj.comm.InterpreterServer.start=false
com.basis.bbj.comm.PortRequestServer.bindAddr=127.0.0.1\:2008
com.basis.bbj.comm.PortRequestServer.maxClients=500
com.basis.bbj.comm.PortRequestServer.start=true
com.basis.bbj.comm.ProxyManagerServer.bindAddr=127.0.0.1\:2009
com.basis.bbj.comm.ProxyManagerServer.maxClients=100
com.basis.bbj.comm.ProxyManagerServer.start=false
com.basis.bbj.comm.RuntimeMgr.notifyUserOnInternalError=false
com.basis.bbj.comm.RuntimeMgr.releaseOnLostConnection=true
com.basis.bbj.comm.RuntimeMgr.requireAllSendMsgResponses=false
com.basis.bbj.comm.RuntimeMgr.suppressPrinterAckBack=true
com.basis.bbj.comm.RuntimeMgr.suppressUIAckBack=true
com.basis.bbj.comm.RuntimeMgr.useDvkLicenseIfPresent=true
com.basis.bbj.comm.SecureThinClientServer.backLog=110
com.basis.bbj.comm.SecureThinClientServer.bindAddr=0.0.0.0\:2103
com.basis.bbj.comm.SecureThinClientServer.disallowConsole=true
com.basis.bbj.comm.SecureThinClientServer.maxClients=500
com.basis.bbj.comm.SecureThinClientServer.start=true
com.basis.bbj.comm.SecureThinClientServer.webUser=rti
com.basis.bbj.comm.TerminalServer.backLog=110
com.basis.bbj.comm.TerminalServer.bindAddr=127.0.0.1\:2004
com.basis.bbj.comm.TerminalServer.maxClients=500
com.basis.bbj.comm.TerminalServer.start=true
com.basis.bbj.comm.ThinClientProxyServer.backLog=110
com.basis.bbj.comm.ThinClientProxyServer.bindAddr=127.0.0.1\:2006
com.basis.bbj.comm.ThinClientProxyServer.err=/usr2/basis/log/ThinClientProxyServer.err
com.basis.bbj.comm.ThinClientProxyServer.keepLogs=7
com.basis.bbj.comm.ThinClientProxyServer.maxClients=500
com.basis.bbj.comm.ThinClientProxyServer.out=/usr2/basis/log/ThinClientProxyServer.out
com.basis.bbj.comm.ThinClientProxyServer.sizeM=10
com.basis.bbj.comm.ThinClientProxyServer.start=false
com.basis.bbj.comm.ThinClientProxyServer.waitTime=0
com.basis.bbj.comm.ThinClientServer.backLog=110
com.basis.bbj.comm.ThinClientServer.bindAddr=0.0.0.0\:2003
com.basis.bbj.comm.ThinClientServer.disallowConsole=true
com.basis.bbj.comm.ThinClientServer.encryptConnection=true
com.basis.bbj.comm.ThinClientServer.maxClients=500
com.basis.bbj.comm.ThinClientServer.start=false
com.basis.bbj.comm.ThinClientServer.translateAppletCmdLine=false
com.basis.bbj.comm.ThinClientServer.translateCmdLine=false
com.basis.bbj.comm.ThinClientServer.webUser=rti
com.basis.bbj.serverPinning=true
com.basis.bbj.sessionPinning=false
com.basis.filesystem.localPort=2000
com.basis.filesystem.localServer=127.0.0.1
com.basis.filesystem.remote.securedataserver.encrypter=com.basis.filesystem.remote.securedataserver.DefaultEncrypter
com.basis.server.LookAndFeel=default
com.basis.server.admin.1.defaultPort=2002
com.basis.server.admin.1.interface=0.0.0.0
com.basis.server.admin.1.start=true
com.basis.server.admin.1.useSSL=false
com.basis.server.admin.log=/usr2/basis/log/AdminServer.log
com.basis.server.allowPipe=true
com.basis.server.autorun.config=/usr2/bbx/config/config.bbx
com.basis.server.autorun.user=root
com.basis.server.autorun.workingDirectory=/usr2/bbx/
com.basis.server.configDirectory=/usr2/bbx/config
com.basis.server.exclusiveAccess=true
com.basis.server.filesystem.1.defaultPort=2000
com.basis.server.filesystem.1.defaultTimeout=2000
com.basis.server.filesystem.1.interface=0.0.0.0
com.basis.server.filesystem.1.prefixes=
com.basis.server.filesystem.1.start=true
com.basis.server.filesystem.1.useSSL=true
com.basis.server.filesystem.log=/usr2/basis/log/FilesystemServer.log
com.basis.server.memStatFrequence=15
com.basis.server.nio.override=false
com.basis.server.passwordAuth=false
com.basis.server.pro5ds.1.64bitMKeyed=false
com.basis.server.pro5ds.1.advisoryLocking=true
com.basis.server.pro5ds.1.defaultPort=2100
com.basis.server.pro5ds.1.defaultTimeout=5
com.basis.server.pro5ds.1.interface=0.0.0.0
com.basis.server.pro5ds.1.maskMKeyedFID=false
com.basis.server.pro5ds.1.prefixes=
com.basis.server.pro5ds.1.scanAllRoots=false
com.basis.server.pro5ds.1.start=false
com.basis.server.pro5ds.1.tagged=false
com.basis.server.pro5ds.1.umask=0000
com.basis.server.pro5ds.1.useSSL=true
com.basis.server.pro5ds.log=/usr2/basis/log/Pro5DSServer.log
com.basis.server.programCacheFile=/usr2/basis/cfg/BBjCache.txt
com.basis.server.programCacheSize=300
com.basis.server.resourceCacheFile=/usr2/basis//cfg/BBjResourceCache.txt
com.basis.server.skipProgramCacheFile=true
com.basis.server.sqlengine.1.debugLevel=2
com.basis.server.sqlengine.1.defaultPort=2001
com.basis.server.sqlengine.1.interface=0.0.0.0
com.basis.server.sqlengine.1.start=true
com.basis.server.sqlengine.1.useSSL=true
com.basis.server.sqlengine.log=/usr2/basis/log/SQLServer.log
com.basis.server.status.defaultPort=11057
com.basis.server.stdErr.keepLogs=7
com.basis.server.stdErr.sizeM=1
com.basis.server.stdErr=/usr2/basis/log/BBjServices.err
com.basis.server.stdOut.keepLogs=7
com.basis.server.stdOut.sizeM=1
com.basis.server.stdOut=/usr2/basis/log/BBjServices.out
com.basis.server.threadPoolSize=-1
com.basis.server.workingDirectory=/usr2/bbx/
com.basis.sql.logLevel=0
com.basis.user.useFork=true
default-config=/usr2/bbx/config/config.bbx
default-user=root
default-working-dir=/usr2/basis/bin
sun.arch.data.model=32
EOF
	close(BBJPROP);
	#Done creating BBj.properties


	#Create BLM install settings file
	my $blm_install_settings_file = "/tmp/blm-install-settings.txt";
	open(BLMINST, "> $blm_install_settings_file");
        print(BLMINST << 'EOF');
-V LICENSE_ACCEPT_BUTTON="true"
-V LICENSE_REJECT_BUTTON="false"
-V IS_SELECTED_INSTALLATION_TYPE=typical
EOF
	close(BLMINST);

	# Basis License Manager.
	showinfo("Installing Basis License Manager...");
	if (! -f "/tmp/$BLM") {
		system("wget --proxy=off --cache=off -O /tmp/$BLM http://$TFSERVER/ks/$BLM");
		if(! -f "/tmp/$BLM") {
			showerror("Could not download file http://$TFSERVER/ks/$BBJ");
		}
	}
	if ( (-f "/tmp/$BLM") && (-f $blm_install_settings_file) ) {
		$bbj_nozip = $BLM;
		$bbj_nozip =~ s/\.gz//g; # blah.jar.gz -> blah.jar
		system("gzip -dc /tmp/$BLM > /tmp/$bbj_nozip");
		system("java -jar /tmp/$bbj_nozip -options $blm_install_settings_file -silent");
	}
	# Basis Admin to setup BBj in general (May need to manually do this)
	showinfo("Running Basis Admin Menu...");
	system("/usr2/basis/bin/admin");

	return(0);
}


sub install_rtiv12_packages
{
	my $if_name = $_[0];

	showinfo("Installing RTI v12 Software Dependencies...");
	showinfo("Using Teleflora Package Server \"$TFSERVER\"");

	# if booting "DHCP", convert to static at specified IP addr
	my $conf_file = "$ALTROOT/etc/sysconfig/network-scripts/ifcfg-$if_name";
	my $pattern = ($OS eq "RHEL6") ? 'BOOTPROTO=\"dhcp\"' : 'BOOTPROTO=dhcp';
	system("grep $pattern $conf_file > /dev/null 2> /dev/null");
	if ($? == 0) {
		showerror("Network must be configured with static ip first");
		return(1);
	}

	remount_filesystem("/teleflora", "/usr2");

	if(! -d "/usr2/bbx/bin") {
		# This step needed to ensure update_ostools succeeds.
		system("mkdir -p /usr2/bbx/bin");
	}

	create_default_users("rti");


	showinfo("Installing Additional Packages...");
	system("yum groupinstall -y mail-server");
	system("yum install -y fetchmail");
	if ($OS ne "RHEL6") {
		system("yum install -y uucp");
	}
	system("yum install -y ksh");


	# Samba
	system("grep delivery /etc/samba/smb.conf > /dev/null 2> /dev/null");
	if($? != 0) {
		modify_samba($POS_TYPE_RTI);
	}

	create_default_samba_users("rti");

	#system("/usr2/bbx/bin/rtiuser.pl --add rti");

	return(0);
}

sub install_daisyv8_packages
{
	my $if_name = $_[0];
	my $dev = "";

	showinfo("Configuring System for Daisy v8+ ...");

	unless ( $KEEPDHCP ) {
	    # if booting "DHCP", convert to static at specified IP addr
	    my $conf_file = "$ALTROOT/etc/sysconfig/network-scripts/ifcfg-$if_name";
	    my $pattern = ($OS eq "RHEL6") ? 'BOOTPROTO=\"dhcp\"' : 'BOOTPROTO=dhcp';
	    system("grep $pattern $conf_file > /dev/null 2> /dev/null");
	    if ($? == 0) {
		showerror("Network must be configured with static ip first");
		return(1);
	    }
	}

	remount_filesystem("/teleflora", "/d");

	# Create a "/mnt/cdrom" item inside of /etc/fstab.
	system("grep \"\/mnt\/cdrom\" /etc/fstab > /dev/null 2> /dev/null");
	if ($? != 0) {
		open(FSTAB, ">> /etc/fstab");
		print(FSTAB "/dev/scd0\t\t/mnt/cdrom\tiso9660\tro,exec\t0 0\n");
		close(FSTAB);
	}

	unless (-d "/mnt/cdrom") {
		system("mkdir /mnt/cdrom");
	}

	unless (-d "/mnt/usb") {
		system("mkdir /mnt/usb");
	}

	# Samba
	system("grep \"/d/daisy/export\" /etc/samba/smb.conf > /dev/null 2> /dev/null");
	if($? != 0) {
		modify_samba($POS_TYPE_DAISY);
	}

	#
	# For daisy systems, since the following function sets the samba password
	# for the daisy account, samba must already be running.
	#
	create_default_users("daisy");
	create_default_samba_users("daisy");

	return(0);
}


sub create_default_users
{
	my $pos = $_[0];
	my $salt = "";
	my $password = "";

	if ($pos eq "rti") {

		system("/usr/sbin/groupadd rti");
		system("/usr/sbin/groupadd rtiadmins");


		# RTI 'tfsupport' system user
		system("grep \"^tfsupport:\" /etc/passwd > /dev/null 2> /dev/null");
		if ($? != 0) {
			system("/usr/sbin/useradd -g rti -G rti,rtiadmins -s /bin/bash tfsupport");
			print("\n");
			print("\n");
			print("\n");
			print(" Set 'tfsupport' User System Password...\n");
			system("passwd tfsupport");
		}


		# RTI 'rti' system user.
		system("grep \"^rti:\" /etc/passwd > /dev/null 2> /dev/null");
		if ($? != 0) {
			$salt = join '', ('.', '/', 0..9, 'A'..'Z', 'a'..'z')[rand 64, rand 64];
			$password = crypt("rti", $salt);
			system("/usr/sbin/useradd -p \"$password\" -g rti -s /bin/bash rti");
		}

		# RTI 'odbc' user for Samba
		system("grep \"^odbc:\" /etc/passwd > /dev/null 2> /dev/null");
		if ($? != 0) {
			system("/usr/sbin/useradd -g rti -s /sbin/nologin odbc");
		}

		# RTI 'delivery' user for Samba
		system("grep \"^delivery:\" /etc/passwd > /dev/null 2> /dev/null");
		if ($? != 0) {
			system("/usr/sbin/useradd -g rti -s /sbin/nologin delivery");
		}
	}


	if ($pos eq "daisy") {

		system("/usr/sbin/groupadd daisy");
		system("/usr/sbin/groupadd dsyadmins");

		# Daisy 'tfsupport' system user.
		system("grep \"^tfsupport:\" /etc/passwd > /dev/null 2> /dev/null");
		if ($? != 0) {
			$salt = join '', ('.', '/', 0..9, 'A'..'Z', 'a'..'z')[rand 64, rand 64];
			$password = crypt("T3l3fl0r4#", $salt);
			system("/usr/sbin/useradd -p \"$password\" -g daisy -G daisy,dsyadmins -s /bin/bash tfsupport");
		}

		# Daisy 'daisy' system user.
		system("grep \"^daisy:\" /etc/passwd > /dev/null 2> /dev/null");
		if ($? != 0) {
			$salt = join '', ('.', '/', 0..9, 'A'..'Z', 'a'..'z')[rand 64, rand 64];
			$password = crypt("dai1sy#", $salt);
			system("/usr/sbin/useradd -p \"$password\" -g daisy -s /bin/bash daisy");
		}
	}
}


sub create_default_samba_users
{
	my $pos = $_[0];
	my $salt = "";
	my $password = "";

	if ($pos eq "rti") {

		# RTI 'rti' Samba user.
		system("grep \"^rti:\" /etc/samba/smbpasswd > /dev/null 2> /dev/null");
		if ($? != 0) {
			showinfo("Adding 'rti' user to samba...");
			open(PIPE, "| smbpasswd -s -a rti");
			print(PIPE "rti\n");
			print(PIPE "rti\n");
			close(PIPE);
		}

		system("grep \"^odbc:\" /etc/samba/smbpasswd > /dev/null 2> /dev/null");
		if ($? != 0) {
			showinfo("Adding 'odbc' user to samba...");
			open(PIPE, "| smbpasswd -s -a odbc");
			print(PIPE "odbc99\n");
			print(PIPE "odbc99\n");
			close(PIPE);
		}

		
		system("grep \"^delivery:\" /etc/samba/smbpasswd > /dev/null 2> /dev/null");
		if ($? != 0) {
			showinfo("Adding 'delivery' user to samba...");
			open(PIPE, "| smbpasswd -s -a delivery");
			print(PIPE "delivery\n");
			print(PIPE "delivery\n");
			close(PIPE);
		}
	}


	if ($pos eq "daisy") {

		# Daisy 'daisy' samba user.
		system("grep \"^daisy:\" /etc/samba/smbpasswd > /dev/null 2> /dev/null");
		if ($? != 0) {
			showinfo("Adding 'daisy' user to samba...");
			open(PIPE, "| smbpasswd -s -a daisy");
			print(PIPE "dai1sy#\n");
			print(PIPE "dai1sy#\n");
			close(PIPE);
		}
	}
}


sub install_apcupsd_conf
{
	#
	# Our default config file will indicate the use of USB based interfaces.
	# Check to see if the conf file has already been customized... if so,
	# nothing to do.
	#
	system("grep updateos /etc/apcupsd/apcupsd.conf > /dev/null 2> /dev/null");
	if ($? == 0) {
		return;
	}

	my $ups_type = "usb";
	if ($UPS_SERIAL) {
		$ups_type = "serial";
	} else {
		$ups_type = identify_ups_hardware();
	}

	# save a copy of the original conf file
	if (-f '/etc/apcupsd/apcupsd.conf') {
		system("cp /etc/apcupsd/apcupsd.conf /etc/apcupsd/apcupsd.conf.orig");
	}

	print("Customizing config file for APC UPS Software...\n");

	open(FILE, "> /etc/apcupsd/apcupsd.conf");

	#
	# The first line of the file must be in a special format or the
	# apcupsd binary will complain.
	#
	print(FILE "## apcupsd.conf v1.1 ##\n");

	print(FILE "#\n");
	print(FILE "# APC UPS Config File\n");
	print(FILE "# Generated by $PROGNAME $CVS_REVISION $TIMESTAMP\n");


	if ($ups_type eq "usb") {
		print(FILE "# USB Based Interface\n");
		print(FILE "UPSCABLE usb\n");
		print(FILE "UPSTYPE usb\n");
		print(FILE "DEVICE \n");

	} else {
		print(FILE "# Serial Based Interface\n");
		print(FILE "UPSCABLE smart\n");
		print(FILE "UPSTYPE smartups\n");
		print(FILE "DEVICE $UPS_SERIAL_PORT\n");
	}

	print(FILE << "xxxEOFxxx");

UPSNAME RTIUPS
LOCKFILE /var/lock
SCRIPTDIR /etc/apcupsd
PWRFAILDIR /etc/apcupsd
NOLOGINDIR /etc
ONBATTERYDELAY 20
BATTERYLEVEL 10
MINUTES 10
TIMEOUT 0
ANNOY 300
ANNOYDELAY 60
NOLOGON disable
KILLDELAY 0

NETSERVER on
NISIP 127.0.0.1
NISPORT 3551
EVENTSFILE /var/log/apcupsd.events
EVENTSFILEMAX 10

UPSCLASS standalone
UPSMODE disable
STATTIME 0
STATFILE /var/log/apcupsd.status
LOGSTATS off
DATATIME 0


xxxEOFxxx
	close(FILE);

	system("chown root:root /etc/apcupsd/apcupsd.conf");
	system("chmod a+r /etc/apcupsd/apcupsd.conf");
}


sub install_apcupsd_battery_script
{
	print("Generating battery script for APC UPS Software...\n");

	# This script is run when we flip to running on the battery.
	open(FILE, "> /etc/apcupsd/onbattery");
	print(FILE "#\n");
	print(FILE  "# APC UPS Config File\n");
	print(FILE  "# Generated by $PROGNAME $CVS_REVISION $TIMESTAMP\n");
	print(FILE << "xxxEOFxxx");
#!/usr/bin/perl
#
# Script to inform Teleflora POS users about being on battery backup (ie, power is out.)
#
use strict;
use warnings;

open(WALL, "| wall");
print(WALL "\n");
print(WALL "\n");
print(WALL "!!!!!!!!!!!!!!!!!!!!!!!\n");
print(WALL "!!!!! POWER OUTAGE !!!!\n");
print(WALL "!!!!! Your Teleflora Server is running on Battery Backup.\n");
print(WALL "!!!!! Please make sure to finish and save any tasks.\n");
print(WALL "!!!!! DO NOT begin critical operations (such as credit cards) until power is restored.\n");
print(WALL "!!!!! Take measures to shut down your server now; before batteries are drained.\n");
print(WALL "!!!!!!!!!!!!!!!!!!!!!!!\n");
print(WALL "\n");
print(WALL "\n");
print(WALL "Press Control-L to refresh the screen.\n");
print(WALL "\n");
close(WALL);

# Magic return value needed by apccontrol.
exit(99);
xxxEOFxxx
	close(FILE);

	system("chown root:root /etc/apcupsd/onbattery");
	system("chmod 555 /etc/apcupsd/onbattery");
}


#
# Install software which listens for UPS notifications (power off, power on.)
# Un-install any previous UPS software (most notably, powerchute under RTI.
#
sub install_apcupsd
{
	#
	# If the rpm is already installed, just deal with config files.
	#
	system("rpm -qa | grep apcupsd > /dev/null 2> /dev/null");
	if($? == 0) {
		showinfo("APC UPS Software Already Installed...");
		install_apcupsd_conf();
		install_apcupsd_battery_script();
		return(0);
	}

	showinfo("Installing APC UPS Software...");

	# remove any old versions previously downloaded
	my $tmp_rpm = "/tmp/apcupsd.rpm";
	system("rm -f $tmp_rpm");

	$OS = determine_os();
	$ARCH = processor_arch();

	my $rpm_name = "";
	if ($OS eq "RHWS5" || $OS eq "RHEL5") {
	    if ($ARCH eq "x86_64") {
		$rpm_name = "apcupsd-3.14.8-1.el5.x86_64.rpm";
	    }
	    else {
		$rpm_name = "apcupsd-3.14.8-1.el5.i386.rpm";
	    }
	}

	elsif ($OS eq "RHEL6") {
	    $rpm_name = "apcupsd-3.14.10-1.el6.x86_64.rpm";
	}

	elsif ($OS eq "RHEL4") {
	    $rpm_name = "apcupsd-3.14.8-1.el4.i386.rpm";
	}

	else {
	    showerror("OS not supported for APC UPS Software: $OS");
	    return(5);
	}

	my $url = "$TFSERVER/rpms/packages/" . $rpm_name;

	# download and install the rpm
	system("cd /tmp && wget -O $tmp_rpm $url");
	system("yum localinstall --nogpgcheck -y $tmp_rpm");

	# verify the rpm was downloaded and installed.
	system("rpm -qa | grep apcupsd > /dev/null 2> /dev/null");
	if ($? != 0) {
		showerror("Can't download and install the APC UPS Software");
		showerror("Teleflora repo: $TFSERVER/rpms/packages");
		showerror("RPM pacakge: $rpm_name");
		return(6);
	}

	# cleanup
	system("rm -f $tmp_rpm");

	unless (-d '/etc/apcupsd') {
		showerror("the APC UPS config dir not found: /etc/apcupsd.");
		showerror("the APC UPS Software was not installed successfully.");
		return(6);
	}

	# Turn off older APC (Java based) UPS Monitor.
	# Note that the APC software did not use 'chkconfig' to install PBEAgent, thus,
	# we will explicitly 'rm' the file.
	if (-f '/etc/init.d/PBEAgent') {
		system("/etc/init.d/PBEAgent stop");
		system("chkconfig --level 3 PBEAgent off");
		system("rm -f /etc/init.d/PBEAgent");
	}

	install_apcupsd_conf();

	install_apcupsd_battery_script();

	system("/sbin/service apcupsd start");

	return(0);
}

sub identify_ups_hardware
{
	my $lsusb_cmd = '/sbin/lsusb';

	unless (-x $lsusb_cmd) {
		return("usb");
	}

	open(PIPE, "$lsusb_cmd |");
	while(<PIPE>) {
		if (/American Power Conversion/i || /Tripp Lite/i) {
			close(PIPE);
			return("usb");
		}
	}
	close(PIPE);

	return("serial");
}


#
# Uninstall the "readahead" rpm.
#
# Red Hat advises removal of RHEL6 readahead package:
#
# "Under some circumstances, the readahead service may cause
# the auditd service to stop. To work around this potential issue,
# disable the readahead collector by adding the following lines
# to the /etc/sysconfig/readahead configuration file:
#   READAHEAD_COLLECT="no"
#   READAHEAD_COLLECT_ON_RPM="no"
# Alternatively, the readahead package can be removed entirely."
#
sub uninstall_readahead
{
	my $redhat_docs = "http://docs.redhat.com/docs/en-US/";
	my $tech_notes = "Red_Hat_Enterprise_Linux/6/html-single/Technical_Notes/index.html";
	my $url = $redhat_docs . $tech_notes;
	my $rpm = "readahead";

	my $exit_status = $EXIT_OK;
	$OS = determine_os();
	if ($OS eq "RHEL6") {
	    showinfo("Removing $OS $rpm rpm per Red Hat tech note...");
	    showinfo("Reference: $url");

	    # attempt to remove it
	    system("rpm -qa | grep -q $rpm");
	    if ($? == 0) {
		system("rpm -e $rpm");
	    }
	    # verify
	    system("rpm -qa | grep -q $rpm");
	    if ($? == 0) {
		showerror("Can't remove $rpm rpm");
		$exit_status = $EXIT_READAHEAD;
	    }
	    else {
		showinfo("$OS $rpm rpm removed");
	    }
	}

	return($exit_status);
}


#
# Initialize the console resolution by editing the kernel line
# in the GRUB config file.
#
# Returns 0 for success, non-zero for error
#
sub init_console_res
{
    my $conf_file = "$ALTROOT/boot/grub/grub.conf";
    my $new_conf_file = "$conf_file.$$";

    unless ($OS eq "RHEL6") {
	showinfo("Initializing console rez supported on RHEL6 only");
	return(0);
    }

    unless (-e $conf_file) {
	showinfo("Can not initialize console rez: $conf_file: does not exist");
	return(0);
    }

    open(OLD, "< $conf_file");
    open(NEW, "> $new_conf_file");

    my $new_res = "video=640x480";
    my $line = "";
    while (<OLD>) {
	    $line = $_;

	    # look for kernel lines
	    if (/^(\s*)kernel(\s+)/) {

		    # change an existing setting or add one if none
		    if (/video=(\S+)/) {
			$line =~ s/video=(\S+)/$new_res/;
		    }
		    else {
			chomp($line);
			$line .= " $new_res\n";
		    }
		    print(NEW $line);
		    next;
	    }
	    print(NEW);
    }
    close(OLD);
    close(NEW);

    if (-s $new_conf_file) {
	system("mv $new_conf_file $conf_file");
	showinfo("GRUB conf file modified to change console rez: $conf_file");
    }
    else {
	system("rm $new_conf_file");
	showerror("Unexpected: new GRUB conf file has zero size: $new_conf_file");
	showerror("GRUB conf file unchanged: $conf_file");
	return($EXIT_GRUB_CONF);
    }

    return(0);
}


#
# Enable verbose boot messages by editing the kernel line
# in the GRUB config file.
#
# Returns 0 for success, non-zero for error
#
sub enable_boot_msgs
{
    my $conf_file = "$ALTROOT/boot/grub/grub.conf";
    my $new_conf_file = "$conf_file.$$";

    unless ($OS eq "RHEL6") {
	showinfo("Enabling boot messages supported on RHEL6 only");
	return(0);
    }

    unless (-e $conf_file) {
	showinfo("Can not enable boot messages: $conf_file: does not exist");
	return(0);
    }

    open(OLD, "< $conf_file");
    open(NEW, "> $new_conf_file");

    my $line = "";
    while (<OLD>) {
	$line = $_;

	# look for kernel lines
	if (/^(\s*)kernel(\s+)/) {

	    # remove "rhgb" from middle of string
	    if ($line =~ / rhgb /) {
		$line =~ s/ rhgb / /;
	    }
	    # remove "rhgb" from end of string
	    elsif ($line =~ / rhgb$/) {
		$line =~ s/ rhgb$//;
	    }

	    # remove "quiet" from middle of string
	    if ($line =~ / quiet /) {
		$line =~ s/ quiet / /;
	    }
	    # remove "quiet" from end of string
	    elsif ($line =~ / quiet$/) {
		$line =~ s/ quiet$//;
	    }
	}

	print(NEW $line);
    }
    close(OLD);
    close(NEW);

    if (-s $new_conf_file) {
	system("mv $new_conf_file $conf_file");
	showinfo("GRUB conf file modified to enable console boot messages: $conf_file");
    }
    else {
	system("rm $new_conf_file");
	showerror("Unexpected: new GRUB conf file has zero size: $new_conf_file");
	showerror("GRUB conf file unchanged: $conf_file");
	return($EXIT_GRUB_CONF);
    }

    return(0);
}


#
# Disable kernel (video) mode setting by editing the kernel line
# in the GRUB config file.
#
# Returns 0 for success, non-zero for error
#
sub disable_kms
{
    my $conf_file = "$ALTROOT/boot/grub/grub.conf";
    my $new_conf_file = "$conf_file.$$";

    unless ($OS eq "RHEL6") {
	showinfo("The --disable-kms option is only supported on RHEL6 platforms");
	showinfo("This platform is: $OS");
	return(0);
    }

    unless (-e $conf_file) {
	showinfo("Can not disable kernel mode setting: $conf_file: does not exist");
	return(0);
    }

    open(OLD, "< $conf_file");
    open(NEW, "> $new_conf_file");

    my $new_string = "nomodeset";
    my $line = "";
    my $lines_changed = 0;
    while (<OLD>) {
	    $line = $_;

	    # look for kernel lines
	    if (/^(\s*)kernel(\s+)/) {

		    # add kernel parameter if it does not exist
		    unless (/nomodeset/) {
			chomp($line);
			$line .= " $new_string\n";
			$lines_changed++;
		    }
		    print(NEW $line);
		    next;
	    }
	    print(NEW);
    }
    close(OLD);
    close(NEW);

    if ($lines_changed == 0) {
	system("rm $new_conf_file");
	showinfo("GRUB conf file unchanged: $conf_file");
    }
    elsif (-s $new_conf_file) {
	system("mv $new_conf_file $conf_file");
	showinfo("GRUB conf file modified to change/add nomodeset: $conf_file");
	showinfo("Number of lines changed: $lines_changed");
    }
    else {
	system("rm $new_conf_file");
	showerror("Unexpected: new GRUB conf file has zero size: $new_conf_file");
	showerror("GRUB conf file unchanged: $conf_file");
	return($EXIT_GRUB_CONF);
    }

    return(0);
}


#
# configure the syslog periodic mark message
#
# For rsyslog on RHEL6:
#
# You must add two directives to the beginning of the conf file;
# the first to load the "immark" module, and the second to set
# the value of "MarkMessagePeriod" to the number of seconds
# between messags.
#
# Example:
#
# $ModLoad immark.so
# $MarkMessagePeriod 1200
#
# The value of "MarkMessagePeriod" specifies how often mark messages are
# to be written to output modules.  The time specified is in seconds.
# Specifying 0 is possible and disables mark messages. In that case,
# however, it is more efficient to NOT load the immark input module.
# Last directive to appear in file wins.  This directive is only
# available after the "immark" input module has been loaded.
#
# For syslog on RHEL5 and previous platforms:
#
# Set the value of the "-m" syslogd option to a non-zero value.  The
# default value is 20, which means 20 minutes between mark messages.
#
# Returns 0 for success, non-zero for error
#
sub modify_syslog_mark
{
    my $syslog_mark_period = $_[0];	# value is in minutes

    if ($syslog_mark_period < 0) {
	showerror("Syslog mark period must be >= 0");
	return(1);
    }

    my $conf_file = ($OS eq "RHEL6") ?
	"$ALTROOT/etc/rsyslog.conf" :
	"$ALTROOT/etc/sysconfig/syslog";
    my $new_conf_file = "$conf_file.$$";

    unless (-e $conf_file) {
	showerror("Syslog conf file not found: $conf_file");
	return(2);
    }

    if ($OS ne "RHEL6") {
	system("grep 'SYSLOGD_OPTIONS=\"-m' $conf_file > /dev/null 2> /dev/null");
	if ($? != 0) {
	    showerror("Can't modify non-standard syslog conf file: $conf_file");
	    showerror("The 'SYSLOGD_OPTIONS' variable is missing or has unexpected format");
	    return(3);
	}
    }

    loginfo("Modifying syslog conf file: $conf_file");

    open(OLD, "< $conf_file");
    open(NEW, "> $new_conf_file");

    if ($OS eq "RHEL6") {
	my $syslog_mark_secs = $syslog_mark_period * 60;
	print(NEW "# BEGIN --syslog-mark lines added by $PROGNAME $CVS_REVISION\n");
	print(NEW "\$ModLoad immark.so\n");
	print(NEW "\$MarkMessagePeriod $syslog_mark_secs\n");
	print(NEW "# END --syslog-mark lines added by $PROGNAME $CVS_REVISION\n");
    }

    while (<OLD>) {
	if ($OS eq "RHEL6") {
	    next if (/^# BEGIN --syslog-mark lines added by $PROGNAME/);
	    next if (/^# END --syslog-mark lines added by $PROGNAME/);
	    next if (/^\$ModLoad\s+immark/);
	    next if (/^\$MarkMessagePeriod\s+\d/);
	}
	else {
	    if (/^(SYSLOGD_OPTIONS=\"-m\s+)(\d+)(.*)$/) {
		print(NEW "${1}$syslog_mark_period${3}\n");
		next;
	    }
	}

	print(NEW);
    }

    close(OLD);
    close(NEW);

    if (-s $new_conf_file) {
	system("mv $new_conf_file $conf_file");
	showinfo("New syslog conf file generated: $conf_file");
	showinfo("Restarting syslog service via the command:");
	my $service_name = ($OS eq "RHEL6") ? "rsyslog" : "syslog";
	showinfo("/sbin/service $service_name restart");
	if ($ALTROOT eq "") {
	    system("/sbin/service $service_name restart");
	}
    }
    else {
	system("rm $new_conf_file");
	showerror("Unexpected: the syslog conf file unchanged: $conf_file");
	return(4);
    }

    return(0);
}


#
# modify the kernel console log message priority
#
# For RHEL5 systems:
#   Edit the configuration file "/etc/sysconfig/syslog"
#   Change the value of the "KLOGD_OPTIONS" variable
#
# For RHEL6 systems:
#   Edit the configuration file "/etc/rsyslog.conf"
#   Load the input module "imklog.so"
#   Set the "$klogConsoleLogLevel" configuration directive
#
# Return 0 for success, non-zero for error
#

sub modify_klog_msg_priority
{
    my $klog_msg_priority = $_[0];

    if ( ($klog_msg_priority < 0) || ($klog_msg_priority > 7) ) {
	showerror("The kernel console message priority must be >= 0 and <= 7");
	return(1);
    }

    my $conf_file = ($OS eq "RHEL6") ?
	"$ALTROOT/etc/rsyslog.conf" :
	"$ALTROOT/etc/sysconfig/syslog";
    my $new_conf_file = "$conf_file.$$";

    unless (-e $conf_file) {
	showerror("Syslog conf file not found: $conf_file");
	return(2);
    }

    if ($OS ne "RHEL6") {
	system("grep 'KLOGD_OPTIONS=' $conf_file > /dev/null 2> /dev/null");
	if ($? != 0) {
	    showerror("Can't modify non-standard syslog conf file: $conf_file");
	    showerror("The 'KLOGD_OPTIONS' variable is missing");
	    return(3);
	}
    }

    loginfo("Modifying syslog conf file: $conf_file");

    open(OLD, "< $conf_file");
    open(NEW, "> $new_conf_file");

    #
    # Always put the modload and configuration directives first in
    # the config file.
    #
    if ($OS eq "RHEL6") {
	print(NEW "# BEGIN --klog-msg-priority lines added by $PROGNAME $CVS_REVISION\n");
	print(NEW "\$ModLoad imklog.so\n");
	print(NEW "\$klogConsoleLogLevel $klog_msg_priority\n");
	print(NEW "# END --klog-msg-priority lines added by $PROGNAME $CVS_REVISION\n");
    }

    while (<OLD>) {

	if ($OS eq "RHEL6") {

	    # remove any old relevant modload and configuration directives
	    next if (/^# BEGIN --klog-msg-priority lines added by $PROGNAME/);
	    next if (/^# END --klog-msg-priority lines added by $PROGNAME/);
	    next if (/^\$ModLoad\s+imklog/);
	    next if (/^\$klogConsoleLogLevel\s+\d/);
	}

	else {

	    # first handle case where KLOGD appears with "-c"
	    if (/^(KLOGD_OPTIONS=\")(.*)(-c\s*\d)(.*\")$/) {
		print(NEW "${1}${2}-c $klog_msg_priority${4}\n");
		next;
	    }
	    # then KLOGD appears but no "-c"
	    elsif (/^(KLOGD_OPTIONS=\")(.*\")$/) {
		print(NEW "${1}-c $klog_msg_priority ${2}\n");
		next;
	    }
	}

	print(NEW);
    }

    close(OLD);
    close(NEW);

    if (-s $new_conf_file) {
	system("mv $new_conf_file $conf_file");
	showinfo("New syslog conf file generated: $conf_file");
	showinfo("Restarting syslog service via the command:");
	my $service_name = ($OS eq "RHEL6") ? "rsyslog" : "syslog";
	showinfo("/sbin/service $service_name restart");
	if ($ALTROOT eq "") {
	    system("/sbin/service $service_name restart");
	}
    }
    else {
	system("rm $new_conf_file");
	showerror("Unexpected: the syslog conf file unchanged: $conf_file");
	return(4);
    }

    return(0);
}


#
# Return 0 for success, non-zero for error
#
sub set_kernel_msg_console
{
    my $kernel_msg_console = $_[0];

    my $conf_file = ($OS eq "RHEL6") ?
	"$ALTROOT/etc/rsyslog.conf" :
	"$ALTROOT/etc/syslog.conf";
    my $new_conf_file = "$conf_file.$$";

    unless (-e $conf_file) {
	showerror("Syslog conf file not found: $conf_file");
	return(2);
    }

    loginfo("Modifying syslog conf file: $conf_file");

    open(OLD, "< $conf_file");
    open(NEW, "> $new_conf_file");
    while (<OLD>) {
	if (/^kern\..*/) {
	    next;
	}
	print(NEW);
    }
    close(OLD);

    print(NEW "kern.*\t\t\t\t\t\t\t$kernel_msg_console\n");

    close(NEW);

    if (-s $new_conf_file) {
	system("mv $new_conf_file $conf_file");
	showinfo("New syslog conf file generated: $conf_file");
	showinfo("Restarting syslog service via the command:");
	my $service_name = ($OS eq "RHEL6") ? "rsyslog" : "syslog";
	showinfo("/sbin/service $service_name restart");
	if ($ALTROOT eq "") {
	    system("/sbin/service $service_name restart");
	}
    }
    else {
	system("rm $new_conf_file");
	showerror("Unexpected: the syslog conf file unchanged: $conf_file");
	return(4);
    }

    return(0);
}


sub configure_kernel_boot_param
{
    my $param_string = $_[0];
    my $param_value = $_[1];

    return(0);
}


sub modify_boot_config
{
    my $mount_point = $_[0];
    my $devname = $_[1];
    my $line = "";

    my $conf_file = "$ALTROOT/boot/grub/grub.conf";
    my $new_conf_file = "$conf_file.$$";

    open(OLD, "< $conf_file");
    open(NEW, "> $new_conf_file");
    while (<OLD>) {
	    $line = $_;

	    #root=LABEL=/ -> root=/dev/blah
	    if(/(root=LABEL=)(\s*)($mount_point)/) {
		    $line =~ s/(root=LABEL=)(\s*)($mount_point)/root=$devname/g;
		    print(NEW $line);
		    next;
	    }
	    print(NEW);
    }
    close(OLD);
    close(NEW);

    system("mv $new_conf_file $conf_file");
}


#
# Modify /etc/fstab
#
# Input:
#   contents of "fs_spec" field (first field)
#   device name to be used in place of current "fs_spec" field
#
# Output:
#   (possibly) rewritten /etc/fstab
#
sub modify_fstab
{
    my $fs_spec = $_[0];
    my $devname = $_[1];
    my $mount_point = $_[2];
    my $fstab_path = "$ALTROOT/etc/fstab";
    my $new_fstab_path = "$fstab_path.$$";
    my @fields = ();
    my $changed = 0;

    open(OLD, "< $fstab_path");
    open(NEW, "> $new_fstab_path");
    while (<OLD>) {
	if (/^(\s*)($fs_spec)(\s+)(.+)/) {
	    @fields = split(/\s+/);
	    print(NEW "$devname\t$mount_point");
	    print(NEW "\t$fields[2]\t$fields[3]\t$fields[4] $fields[5]\n");
	    $changed = 1;
	    next;
	}
	print(NEW);
    }
    close(OLD);
    close(NEW);

    # If the fstab was changed, replace old version with new.
    # Else, removed new one.
    if ($changed) {
	system("mv $new_fstab_path $fstab_path");
    }
    else {
	system("rm $new_fstab_path");
    }
}


#
# Given the "fs_spec" field from the fstab in the form of either
# "LABEL=string" or "UUID=string", match that to the output of
# the blkid command and convert to a device name.
#
# Note, the fstab field does not have the parameter value in QUOTES
# while the blkid does.
#
# Input:
#	file system id in the form of either LABEL=string or UUID=string
#
# Returns:
#	the block special device name
#
sub filesystem_id_to_devname
{
    my $id = $_[0];
    my $devname = "";

    my $cmd = '/sbin/findfs';
    system("$cmd $id > /dev/null 2> /dev/null");
    if ($? == 0) {
	$devname = qx($cmd $id);
	chomp $devname;
    }
    else {
	showerror("Could not convert filesys id to devname: $cmd returned non-zero exit status");
    }

    loginfo("Filesystem id $id converted to devname: $devname");

    return($devname);
}


#
# Given the mount point, Get the "fs_spec" field from the fstab 
#
# Input:
#	the "file system mount point", ie the second field of an fstab entry,
#	named the "fs_file" field
#
# Return:
#	return the "block special device name", ie the first field of the
#	fstab entry, the "fs_spec" field.
#
# Note, the "block special device name" could be one of several forms, even an
# actual device name but what we are looking for is:
#
# 1) LABEL=/teleflora
# 2) UUID=d78cf896-6f1a-4bb7-a2d7-71e4c2ee7d18
#
sub get_fstab_info
{
    my $mount_point = $_[0];
    my $fs_spec = "";
    my $uuid = "";
    my @fstab_entry = ();

    my $fstab_path = "$ALTROOT/etc/fstab";
    open(PIPE, "cat $fstab_path |");
    while(<PIPE>) {

	# skip comments or blank lines
	next if ( (/^(\s*)#/) || (/^(\s*)$/) );

	# break line into fields
        @fstab_entry = split(/\s+/);
	next unless (@fstab_entry);

	if ($mount_point eq $fstab_entry[1]) {
	    $fs_spec = $fstab_entry[0];
	    last;
	}

    }
    close(PIPE);

    return($fs_spec);
}


#
# By default, RHEL5 mounts file systems based on disk label and
# RHEL6 mounts file systems  based on UUID.
#
# Input:
#   current mount point of the file system
#   desired new mount point of the file system
#
# Returns:
#   0 for success
#   non-zero for error
#
# Side effects:
#   This function modifies /etc/fstab to remove Label based or
#   UUID based mounting for RHEL5 and RHEL6 respectively and
#   instead, uses block special device names in the first field
#   of the fstab.
#
sub remount_filesystem
{
    my $mount_point = $_[0];
    my $new_mount_point = $_[1];
    my $devname = "";
    my $fs_spec = "";

    if ($mount_point ne "/") {
	my $cwd = getcwd();
	if ($cwd =~ /^$mount_point/) {
	    showerror("The cwd resides within the file system to be remounted: $mount_point");
	    print "Please change directory to outside of that file system\n";
	    return(2);
	}
    }

    # get fs_spec field (first field) from fstab
    $fs_spec = get_fstab_info($mount_point);
    if ($fs_spec eq "") {
	logerror("Mount point <$mount_point>: not found in /etc/fstab");
	return(1);
    }

    # if the fs_spec field is in form of "LABEL=" or "UUID=", then use
    # the blkid command to convert to device name
    if ( ($fs_spec =~ /LABEL=/) || ($fs_spec =~ /UUID=/) ) {
	$devname = filesystem_id_to_devname($fs_spec);
    }
    else {
	$devname = $fs_spec;
    }

    showinfo("Re-Mounting: old: $fs_spec mounted on $mount_point");
    showinfo("Re-Mounting: new: $devname mounted on $new_mount_point");

    if ($new_mount_point ne "/") {
	system("umount $mount_point");
	if ($? != 0) {
	    logerror("Warning: Could not umount <$mount_point>");
	}
    }

    showinfo("Re-Writing fstab: changing entry with file system id $fs_spec to device $devname");
    modify_fstab($fs_spec, $devname, $new_mount_point);

    # In the case of our root partition, we also need to modify grub.conf
    if ($mount_point eq "/") {
	modify_boot_config($mount_point, $devname);
    }

    # Make the new mount points if necessary.
    if( ("$new_mount_point" ne "swap") && (! -d "$new_mount_point") ) {
	system("mkdir -p $new_mount_point");
    }

    if ("$new_mount_point" ne "/") {
	system("mount $new_mount_point");
    }

    return(0);
}


#
# Input:
#	content type the block device holds
#
# Returns:
#	the LABEL or UUID field and the device name field
#
sub get_blkid_info
{
    my $type = $_[0];
    my $blkid_cmd = '/sbin/blkid';

    unless (open(PIPE, "$blkid_cmd |")) {
	showerror("Could not remount swap partition: could not run $blkid_cmd");
	return(2);
    }

    # get the device name and the "fs_spec" from the output of blkid
    my @fields = ();
    my $devname = "";
    my $fs_spec = "";
    my $content_type = "TYPE=\"$type\"";
    while(<PIPE>) {
	next until /$content_type/;

	@fields = split(/:/);
	$devname = $fields[0];

	@fields = split(/\s+/, $fields[1]);
	foreach my $field (@fields) {
	    if ( ($field =~ /LABEL=".+"/) || ($field =~ /UUID=".+"/) ) {
		$fs_spec = $field;
		last;
	    }
	}

	last;
    }
    close(PIPE);

    my @info_list = ($fs_spec, $devname);
    return(@info_list);
}


#
# The output of the blkid(8) command has everything we need...
# each line will be:
#
#   device name: parameter="value"...
#
# if there is a label, the line will have:
#   LABEL="label"
# if there is a UUID, the line will have:
#   UUID="uuid"
# for the swap partition, the line will have:
#   TYPE="swap"
#
sub remount_swap
{
    my $fs_spec = "";
    my $devname = "";

    # The "blkid" command provides info about the swap device:
    # 1) the fstab "fs_spec" field
    # 2) the device name
    ($fs_spec, $devname) = get_blkid_info("swap");

    if ( ($devname eq "") || ($fs_spec eq "") ) {
	logerror("Could not remount swap with device name instead of filesystem id");
	return("");
    }

    # first, disable swap
    loginfo("Disabling swap");
    system("swapoff $devname");

    #
    # At this point, the fs_spec field looks like:
    #
    #	LABEL="label" or UUID="uuid"
    #
    # so we need to get rid of the quotes because fstab entries do not
    # use quotes around the value.
    #
    $fs_spec =~ s/"//g;

    # edit the fstab entry
    modify_fstab($fs_spec, $devname, "swap");

    # now, re-enable swap - it's now using a device in the
    # fstab instead of a label (RHEL5) or a UUID (RHEL6).
    loginfo("Enabling swap");
    system("swapon $devname");

    return($devname);
}


sub reset_aide_database
{

	if(! -f "/usr/sbin/aide") {
		return(1);
	}

	showinfo("Resetting Intrusion Detection Rules...");

	if(-f "/var/lib/aide/aide.db.gz") {
                $PID = fork();
                if ($PID == 0) {
                        loginfo("Update Aide Databases.");
                        system("/usr/sbin/aide --init");
                        system("mv /var/lib/aide/aide.db.new.gz /var/lib/aide/aide.db.gz");
                        exit(0);
                } else {
                        loginfo("Update Aide Database started as PID $PID at " . localtime() . "\n");
                }
                exit(0);
        }

	return(0);
}





#
# Unmount all filesystems except '/'
# Useful if we need to perform a "power button" shutdown, to ensure that most of our 
# filesystems don't come up in a 'dirty' state.
#
# Notes:
# We explicity do not run 'sync' as, I have experienced in the past, cases where 'sync' hung,
# I believe because the backup device could not sync, and, "sync" works on all devices.
# The filesystem caches are all sycned when the FS is unmounted.
# This function should be run as root.
#
sub hard_reboot
{
	my @array = ();
	my $mountpoint = "";
	my $device = "";
	my $pid = -1;
	my $childpid = -1;
	my $i = 0;


	showinfo("Performing reboot...");

	# Run the "regular" shutdown here. If that works as expected, we should never actually
	# finish this function, as, we will be killed off.
	$pid = fork();
	if($pid == 0) {
		exec("/sbin/shutdown -r now");
	}
	for($i = 0; $i < 120; $i++) {
		$childpid = waitpid($pid, WNOHANG);
		last if($childpid != 0);
		sleep(1);
	}
	if($childpid < 0) {
		showinfo("waitpid() on PID $pid returned $childpid.");
	}

	if($i >= 120) {
		showerror("We waited long enough (2 mins) for our shutdown to finish. Will manually reboot.");
		system("kill -TERM $pid");
	} else {
		showerror("the 'shutdown' sub process terminated in $i seconds, without terminating this process.");
		return(1);
	}

	# If we get this far, though "shutdown" ran, it never actually did anything.
	# We have seen this in the field in cases where sync() hangs (and hence, the 'shutdown' process 
	# also hangs.
	# At this point, we will manually unmount disks, kill off processes, and reboot the computer.
	# Note! Make sure *not* to run 'sync', as, that will almost certainly hang.
	showinfo("Server did not reboot normally. Will issue a 'hard' reboot...");
	sleep(5);

	# Bring down
	system("/sbin/service smb stop");
	system("/sbin/service sendmail stop");
	if (-f "/etc/init.d/rti") {
		system("/sbin/service rti stop");
	}
	if(-f "/etc/init.d/daisy") {
		system("/sbin/service daisy stop");
	}


	# If things are this un-healthy, then, we may have underlying disk issues.
	# If we simply reboot without the '/fsckoptions' file, then, chances are, the flower shop
	# owner will be presented with an ugly prompt saying 'fsck failed', and a '#' prompt.
	# By adding this particular file, according to /etc/rc.d/rc.sysinit, FSCK will use these
	# "force" and "dammit" options, thereby sparing the florist from having to type stuff in.
	open(FILE, "> /fsckoptions");
		print(FILE "-f -v -y");
	close(FILE);

	open(MOUNTS, "< /proc/mounts");
	while(<MOUNTS>) {
		chomp;
		next until(/ext[23]/);
		@array = split(/\s+/);
		$device = $array[0];
		$mountpoint = $array[1];
		next if(! $device);
		next if("$device" eq "");
		next if("$mountpoint" eq "");
		next if("$mountpoint" eq "/");
		next if("$mountpoint" eq "/proc");
		next if(! -b $device);

		# Look at all processes holding this filesystem open, and kill them.
		#[kachline@rtidev bin]$ sudo /usr/sbin/lsof /usr4
		#COMMAND  PID USER   FD   TYPE DEVICE SIZE   NODE NAME
		#bash    4283   jj  cwd    DIR  104,2 4096 961291 /usr4/jj/rev
		print("Killing Processes ($device)...\n");
		open(LSOF, "/usr/sbin/lsof $device |");
		while(<LSOF>) {
			chomp;
			@array = split(/\s+/);
			$pid = $array[1];
			next if(! $pid);
			next if("$pid" eq "PID");
			print("$_\n");
			system("kill -HUP $pid");
			system("kill -TERM  $pid");
			system("kill -INT $pid");
			system("kill -KILL $pid");
		}
		close(LSOF);

		# Unmount the filesystem
		print("Unmounting $mountpoint -> $device");
		system("umount $device");
		if($? != 0) {
			print("Forcing Unmount of $mountpoint -> $device");
			system("umount -f $device");
			if($? != 0) {
				print("Error: Could not unmount $mountpoint -> $device");
				print("Could not Unmount \"$device\".\n");
				print("Try re-running this script to kill off 'lsof' processes.\n");
			}
		}

	}
	close(MOUNTS);

	#
	# Try to do a standard reboot at this point, though we have seen
	# instances in the field where this does not work.
	#
	system("/sbin/reboot -n -f");

	#
	# This is the atom bomb for causing a reboot...
	#
	# For background on what the code below does, read this article:
	#	http://www.linuxjournal.com/content/rebooting-magic-way
	#
	system("echo 1 > /proc/sys/kernel/sysrq");
	system("echo b > /proc/sysrq-trigger");

	#
	# Should not reach this point... the system should have rebooted
	# before this.
	#

	return(2);
}



#
# Many of our original T300 servers came shipped with a swap partition in place, yet,
# a bug in /etc/fstab which would cause the swap partition to never "swapon".
# This routing fixes said bug.
#
sub swapon
{
	my @partitions = ();
	my @array = ();
	my $thisdisk = "";
	my $thispart = "";


	showinfo("Enabling Swap Space...");
	# only RTI systems
	if(-d "/d/daisy") {
		showinfo("This is a daisy system. Will not update swap space.");
		return (0);
	}


	# Are we root?
	if($UID != 0) {
		showerror("Must be root to run.");
		return(1);
	}


	# Which partitions are formatted for swap?
	foreach $thisdisk ("/dev/sda", "/dev/sdb") {
		open(FDISK, "/sbin/fdisk -l $thisdisk |");
		while(<FDISK>) {
			next until(/^\/dev\//);
			next unless(/(Linux)(\s+)(swap)/);
			@array = split(/\s+/);
			$thispart = $array[0];
			if(-b "$thispart") {
				push(@partitions, $thispart);
			}
		}
		close(FDISK);
	}


	# Did we find swap partitions?
	if($#partitions < 0) {
		showerror("No swap partitions were found! No changes made.");
		return(5);
	}


	# Modify /etc/fstab
	showinfo("Updating /etc/fstab...");
	open(OLD, "< /etc/fstab") or die("Could not open /etc/fstab for read");
	open(NEW, "> /etc/fstab.$$") or die("Could not open /etc/fstab.$$ for write");
	while(<OLD>) {
		next if(/(\s+)(swap)(\s+)/);
		print(NEW);
	}
	close(OLD);

	# Record our known swap partitions.
	foreach $thispart (sort(@partitions)) {
		print(NEW "$thispart\t\tswap\t\tswap\tdefaults\t\t0\t0\n");
	}
	close(NEW);

	if(-s "/etc/fstab.$$" <= 0) {
		# We probably didn't have permission to create this temp file.
		showerror("Our newly created fstab file does not exist. No changes made.");
		return(6);
	}
	system("mv /etc/fstab.$$ /etc/fstab");


	# Format each swap partition.
	foreach $thispart (sort(@partitions)) {
		showinfo("Enabling Swap Partition \"$thispart\" ...");
		system("/sbin/mkswap $thispart");
		system("/sbin/swapon $thispart");
	}
	close(NEW);

	# Enable our swap partitions.
	system("cat /proc/meminfo | grep Swap");

	showinfo("Swap Space Updated.");
	return(0);
}


#
# Remove files from /var/spool/cups/tmp
# Earlier versions of RTI would leave behind lots of zero byte files in 
# this directory.
#
sub cups_clean_tempfiles
{
	showinfo("Cleanup CUPS Temporary Files...");
	system("find /var/spool/cups/tmp -type f -size 0b -print | xargs rm -f");
	if(-d "/usr2/bbx/bbxtmp") {
		system("find /usr2/bbx/bbxtmp -type f -size 0b -print | xargs rm -f");
	}
	return(0);
}


#
# Edit the CUPS conf file.
#
sub cups_edit_conf
{
	my $conf_file = '/etc/cups/cupsd.conf';
	my $new_conf_file = "$conf_file-$TIMESTAMP";

	showinfo("Editing CUPS Configuration File...");

	unless (-f "$conf_file") {
		showerror("$conf_file not found");
		return(1);
	}

	unless (open(OLD, "< $conf_file")) {
		showerror("Could not open $conf_file for read");
		return(2);
	}

	unless (open(NEW, "> $new_conf_file")) {
		showerror("Could not open $new_conf_file for write");
		close(OLD);
		return(3);
	}

	while(<OLD>) {

		next if (/^#updateos.pl#/);

		# CUPS has a default "max jobs" set to 500. For some of the
		# larger shops, this print queue size is too small.
		# Set the "Max Jobs" setting to a higher value.

		if ($CUPSCONF_MAXJOBS) {
			if(/^(\s*)(MaxJobs)(\s+)/) {
				print(NEW "#updateos.pl# $_");
				next;
			}
		}


		#
		# Don't put files into /var/spool/cups/tmp at all.
		# Thus eliminating the need for the "cleanfiles"
		# patch elsewhere in this script.
		#
		if ($CUPSCONF_DISABLETEMPFILES) {
			if (/^(\s*)(PreserveJobHistory)(\s+)/i) {
				print(NEW "#updateos.pl# $_");
				next;
			}
			if (/^(\s*)(PreserveJobFiles)(\s+)/i) {
				print(NEW "#updateos.pl# $_");
				next;
			}
		}

		if ($CUPSCONF_TIMEOUTS) {
			if (/^(\s*)(Timeout)(\s+)/i) {
				print(NEW "#updateos.pl# $_");
				next;
			}
		}

		if ($CUPSCONF_ERRORPOLICY) {
			if (/^(\s*)(ErrorPolicy)(\s+)/i) {
				print(NEW "#updateos.pl# $_");
				next;
			}
		}
		print(NEW);
	};

	close(OLD);

	loginfo("Adding rules to $conf_file");
	print(NEW "#updateos.pl# --- Start of additional lines ---\n");
	print(NEW "#updateos.pl# $PROGNAME $CVS_REVISION $TIMESTAMP\n");
	print(NEW "MaxJobs 10000\n");
	print(NEW "PreserveJobHistory No\n");
	print(NEW "PreserveJobFiles No\n");

	determine_os();

	if ($OS eq "RHEL6") {
	    print(NEW "Timeout $CUPSCONF_RHEL6_TIMEOUT\n");
	}
	else {
	    print(NEW "Timeout $CUPSCONF_RHEL5_TIMEOUT\n");
	}

	if ($OS ne "FC3") {
		print(NEW "ErrorPolicy retry-job\n");
	}
	print(NEW "#updateos.pl# --- End of additional lines ---\n");

	close(NEW);

	# If we created a new conf file that is zero sized, that is bad.
	if (-z $new_conf_file) {
		showerror("Copy of $conf_file is a zero size file");
		system("rm $new_conf_file");
		return(4);
	}

	# Assume conf file was successfully transformed...
	# so replace the old one with the new.

	system("chmod --reference=$conf_file $new_conf_file");
	system("chown --reference=$conf_file $new_conf_file");
	system("cp $new_conf_file $conf_file");

	system("/sbin/service cups start");

	return(0);
}


#
# Remove all print jobs from all print queues.
#
sub purge_printjobs
{
	showinfo("Purging CUPS Print Jobs...");
	system("/sbin/service cups stop");
	system("find /var/spool/cups -type f -print | xargs rm -f");
	system("/sbin/service cups start");
	return(0);
}




sub determine_os
{
	$OS = "";

	# We could also use /usr/bin/lsb-release
	open(FILE, "< /etc/redhat-release");
	while(<FILE>) {

		# Fedora Core 3
		if(/(Fedora)(\s+)(Core)(\s+)(release)(\s+)(3)/) {
			$OS = "FC3";
			last;
		}

		# Fedora Core 4
		if(/(Fedora)(\s+)(Core)(\s+)(release)(\s+)(4)/) {
			$OS = "FC4";
			last;
		}

		# Fedora Core 5
		if(/(Fedora)(\s+)(Core)(\s+)(release)(\s+)(5)/) {
			$OS = "FC5";
			last;
		}

		# Redhat Enterprise Linux Client Workstation 5
		if( /(Client release)(\s+)(5)/ ) {
			$OS = "RHWS5";
			last;
		}

		# ES 6
		# Redhat Enterprise Linux Server 6
		if( /(release)(\s+)(6)/ ) {
			$OS = "RHEL6";
			last;
		}

		# ES 5
		# Redhat Enterprise Linux Server 5
		if( (/(release)(\s+)(5)/)
		||  (/(CentOS)([[:print:]]+)(\s)(5)/) ) {
			$OS = "RHEL5";
			last;
		}

		# EL 4
		# Redhat Enterprise Linux Server 4
		if( (/(release)(\s+)(4)/) 
		||  (/(CentOS)([[:print:]]+)(\s)(4)/) ) {
			$OS = "RHEL4";
			last;
		}

		# EL 3
		# Redhat Enterprise Linux Server 3
		if (/(release)(\s+)(3)/) {
			$OS = "RHEL3";
			last;
		}

		# Redhat 7.2
		if(/(release)(\s+)(7\.2)/) {
			$OS = "RH72";
			last;
		}
	}
	close(FILE);

	if ($OS eq "") {
	    $OS = "unsupported";
	}

	#
	# Allow a last minute switcheroo.
	#
	if ($ALTOS) {
	    $OS = $ALTOS;
	}

	return($OS);
}


sub get_arch_info
{
        my $arch = "";

        open(PIPE, "uname -i |");
        while(<PIPE>) {
                if(/i386/) {
                        $arch = "i386";
                }
                if(/x86_64/) {
                        $arch = "x86_64";
                }
        }
        close(PIPE);

        return($arch);
}


#
# Which processor architecture are we running on?
#
sub processor_arch
{
	$ARCH = "";

        $ARCH = get_arch_info();

        if ($ARCH eq "") {
                showerror("Can't determine processor architecture - assuming: i386");
                $ARCH = "i386";
        }

	return($ARCH);
}

sub report_rhn_system_id
{
	my $rhn_system_id = get_rhn_system_id();

	if ($rhn_system_id eq "") {
		$rhn_system_id = "unregistered";
	}
	print("Redhat Network System ID: $rhn_system_id\n");
	return(0);
}


#
# get the Red Hat Network system id
#
# Returns
#   rhn system id on success
#   empty string if not registered
#
sub get_rhn_system_id
{
	my $rhn_system_id = "";
	my $rhn_system_id_file = '/etc/sysconfig/rhn/systemid';

	#
	# If the rhn system id file does not exist, then the system is
	# not registered.
	#
	unless (-f $rhn_system_id_file) {
		return($rhn_system_id);
	}

	open(FILE, "< $rhn_system_id_file");
	while (<FILE>) {
		if (/^<name>system_id/) {
			my $next_line = <FILE>;
			if ($next_line =~ /^<value><string>(.+)<\/string><\/value>/) {
				$rhn_system_id = $1;
				last;
			}
		}
	}
	close(FILE);

	return($rhn_system_id);
}


sub get_sys_info
{
	my $arch = "";

	return("x86") if ($OS eq 'RH72');

	open(PIPE, "uname -i |");
	while(<PIPE>) {
		if(/i.86/) {
			$arch = "x86";
		}
		if(/x86_64/) {
			$arch = "x64";
		}
	}
	close(PIPE);

	return($arch);
}


# strip domain name from fully qualified domain name if present.
#
# Returns
#   hostname
#
sub strip_domain
{
    my ($hostname, $domain) = @_;

    if ($hostname =~ /(.+)\.$domain/) {
	$hostname =~ s/\.$domain//;
    }

    return($hostname);
}


sub get_hostname
{
    my $hostname = hostname();

    return($hostname);
}


sub get_network_attribute
{
    my ($device, $selector) = @_;

    my $attribute_val = "";
    open(PIPE, "/sbin/ifconfig $device 2> /dev/null |");
    while(<PIPE>) {

	my $pattern = "";
	if ($selector eq $NET_ATTR_IPADDR) {
	    $pattern = "inet addr";
	}
	elsif ($selector eq $NET_ATTR_BROADCAST) {
	    $pattern = "Bcast";
	}
	elsif ($selector eq $NET_ATTR_NETMASK) {
	    $pattern = "Mask";
	}

	if (/($pattern):(\d+\.\d+\.\d+\.\d+)/) {
	    $attribute_val = $2;
	}
    }
    close(PIPE);

    return($attribute_val);
}


#
# change the value of the "HOSTNAME" variable in the sysconfig network
# file - normally "/etc/sysconfig/network" but could be different when
# testing.
#
# Returns
#   1 on success
#   0 on error
#
sub set_sysconfig_hostname
{
    my ($hostname, $config_file_path) = @_;

    my $rc = 1;

    system("perl -p -i -e 's/^HOSTNAME=.*/HOSTNAME=$hostname/' $config_file_path");
    if ($? != 0) {
	$rc = 0;
    }

    return($rc);
}


sub update_hosts_config_file
{
    my ($new_hostname, $ipaddr, $config_file_path) = @_;

    my $rc = 1;

    my $new_config_file_path = $config_file_path . '.' . "$$";

    my $current_hostname = get_hostname();

    # form a new line suitable for the /etc/hosts file
    my $fqdn = $new_hostname . ".teleflora.com";
    my $hosts_file_entry = "$ipaddr\t$fqdn $new_hostname\n";

    if (open(my $ocfh, '<', $config_file_path)) {
        if (open(my $ncfh, '>', $new_config_file_path)) {

	    my $hostname_replaced = 0;

	    while (<$ocfh>) {

		# if the loopback line:
		#   remove current hostname if it appears
		# else if line contains old hostname
		#   update current hostname
		# else
		#   pass through
		if (/^127\.0\.0\.1/) {
		    if ($current_hostname ne "localhost") {
			if (/$current_hostname/) {
			    s/$current_hostname(\s*)//;
			}
		    }
		}
		elsif (/$current_hostname/) {
		    $_ = $hosts_file_entry;
		    $hostname_replaced = 1;
		}

		print($ncfh $_);
	    }

	    close($ocfh);

	    # if the current hostname was not replaced,
	    # which means it did not appear in the /etc/hosts file, then
	    # just append a new entry now.
	    unless ($hostname_replaced) {
		print($ncfh $hosts_file_entry);
	    }

	    close($ncfh);
	}
	else {
	    $rc = 0;
	}
    }
    else {
	$rc = 0;
    }

    if ($rc) {
	system("mv $new_config_file_path $config_file_path");
    }

    return($rc);
}


#
# Gather some evidence if we believe there was a compromise or need performance
# statistics.
#
sub forensics
{
	my $pid = "";
	my @array = ();


	unlink("/tmp/system-info.$$");

	system("/bin/echo '----- Basic Information -----' | tee --append /tmp/system-info.$$");
	system("date >> /tmp/system-info.$$");
	system("hostname >> /tmp/system-info.$$");


	system("/bin/echo '----- Network Information -----' | tee --append /tmp/system-info.$$");
	system("/sbin/ifconfig >> /tmp/system-info.$$");
	system("/bin/netstat -plan >> /tmp/system-info.$$");
	system("/sbin/route -n >> /tmp/system-info.$$");

	#
	# Run nmap discovery on all networks which we have a (known) route to.
	#
	open(PIPE, "/sbin/route -n |");
	while(<PIPE>) {
		if(/^\d/) {
			@array = split(/(\s+)/);
			next if("$array[0]" eq "0.0.0.0");
			next if("$array[0]" eq "127.0.0.0");
			next if("$array[0]" !~ /^\d/);
			system("/bin/echo '----- nmap $array[0] -----' | tee --append /tmp/system-info.$$");
			system("nmap -v -n -sS -O $array[0]/24 >> /tmp/system-info.$$");
		}
	}
	close(PIPE);


	system("/bin/echo '----- Processes (ps wwaux) -----' | tee --append /tmp/system-info.$$");
	system("ps wwaux >> /tmp/system-info.$$");

	# We may not be able to trust 'ps' results.
	system("/bin/echo '----- Processes (/proc) -----' | tee --append /tmp/system-info.$$");
	open(PROCS, "find /proc -type d -print |");
	while(<PROCS>) {
		chomp;
		next until (/\d+/);
		$pid = $_;
		print("$pid: ");
		system("/bin/echo \"/proc/$pid/cmdline\" >> /tmp/system-info.$$");
		system("cat /proc/$pid/cmdline >> /tmp/system-info.$$");
	}
	close(PROCS);


	system("/bin/echo '----- System Performance -----' | tee --append /tmp/system-info.$$");
	system("cat /proc/meminfo >> /tmp/system-info.$$");
	system("vmstat 2 5 >> /tmp/system-info.$$");
	system("iostat -x 2 5 >> /tmp/system-info.$$");
	system("/usr/sbin/dmidecode >> /tmp/system-info.$$");


	system("/bin/echo '----- OS Information -----' | tee --append /tmp/system-info.$$");
	system("uname -a >> /tmp/system-info.$$");
	system("cat /proc/redhat-release >> /tmp/system-info.$$");
	system("rpm --list >> /tmp/system-info.$$");
	system("rpm --verify >> /tmp/system-info.$$");
	system("dmesg >> /tmp/system-info.$$");
	if(-f "/usr/bin/yum") {
		system("yum checkupdate >> /tmp/system-info.$$");
	}



	system("/bin/echo '----- User Information -----' | tee --append /tmp/system-info.$$");
	system("w >> /tmp/system-info.$$");
	system("last >> /tmp/system-info.$$");
	system("/usr2/bbx/bin/rtiuser.pl --list >> /tmp/system-info.$$");

	system("/bin/echo '----- PCI 2.2.2 -----' | tee --append /tmp/system-info.$$");
	system("/sbin/chkconfig --list | grep ':on' >> /tmp/system-info.$$");

	system("/bin/echo '----- PCI 2.2.3 -----' | tee --append /tmp/system-info.$$");
	system("/sbin/chkconfig --list | grep ':on' >> /tmp/system-info.$$");



	# Create our resultant tarfile.
	unlink("/tmp/system-info.$$.tar");
	system("cd /tmp && tar --append -vf /tmp/system-info.$$.tar system-info.$$");
	system("cd / && tar --append -vf /tmp/system-info.$$.tar /etc");
	system("cd /home && find . -type f -iname .bash_history -exec tar --append -vf /tmp/system-info.$$.tar \\{\\} \\;");
	system("cd / && find /root -type f -iname .bash_history -exec tar --append -vf /tmp/system-info.$$.tar \\{\\} \\;");
	system("cd / && tar --append -vf /tmp/system-info.$$.tar root");
	system("cd / && tar --append -vf /tmp/system-info.$$.tar var/log");
	if(-d "/usr2/bbx") {
		system("cd / && tar --append -vf /tmp/system-info.$$.tar usr2/bbx/log");
		system("cd / && tar --append -vf /tmp/system-info.$$.tar usr2/bbx/config");
	}
	if(-d "/d/daisy") {
		system("cd / && tar --append -vf /tmp/system-info.$$.tar d/daisy/log");
	}

	system("gzip -v9 /tmp/system-info.$$.tar");
	system("ls -la /tmp/system-info.$$.tar.gz");


	return(0);
}



sub showinfo
{
	my $message = $_[0];
	print("$message\n");
	return(loginfo($message));
}

sub loginfo
{
	my $message = $_[0];
	return(logit($message, 'I'));
}

sub showerror
{
	my $message = $_[0];
	print("Error: $message\n");
	return(logerror($message));	
}

sub logerror
{
	my $message = $_[0];
	return(logit($message, 'E'));
}

sub logit
{
	my $message = $_[0];
	my $type = $_[1];
	my $logtime = strftime("%Y-%m-%d %H:%M:%S", localtime());

	chomp($message);

	# where is the logfile?
	if ($LOGFILE_PATH eq "") {
		if ( ($POSDIR ne "") && (-d "$POSDIR/log") ) {
			$LOGFILE_PATH = "$POSDIR/log/RTI-Patches.log";
		} else {
			$LOGFILE_PATH = "/tmp/RTI-Patches.log";
		}
	}

	open LOGFILE, ">> $LOGFILE_PATH";
		print LOGFILE "$logtime ($PROGNAME-$$) <$type> $message\n";
	close LOGFILE;

	return "";
}


__END__

=pod

=head1 NAME

updateos.pl - Teleflora Operating System Updater


=head1 VERSION

This documenation refers to version: $Revision: 1.268 $


=head1 USAGE

updateos.pl

updateos.pl B<--version>

updateos.pl B<--help>

updateos.pl B<--verbose>

updateos.pl B<--java>

updateos.pl B<--java-version=s>

updateos.pl B<--rti14>

updateos.pl B<--syslog-mark=n>

updateos.pl B<--kernel-msg-console=n>

updateos.pl B<--samba-gen-conf>

updateos.pl B<--samba-set-passdb>

updateos.pl B<--samba-rebuild-passdb>

updateos.pl B<--init-console-res>

updateos.pl B<--enable-boot-msgs>

updateos.pl B<--disable-kms>

updateos.pl B<--cupsconf>


=head1 OPTIONS

=over 4

=item B<--version>

Output the version number of the script and exit.

=item B<--help>

Output a short help message and exit.

=item B<--verbose>

For some operations, output more information.

=item B<--java>

Install the latest version of the Oracle Java JRE package.

=item B<--java-version=s>

Install the version of the Oracle Java JRE package specified.

=item B<--rti14>

Configure the system appropriate for running RTI version 14.

=item B<--syslog-mark=n>

Configure the syslog mark message period.

=item B<--kernel-msg-console=string>

Configure syslog to direct kernel messages to specified tty device.

=item B<--samba-gen-conf> or B<--samba>

Generate a Samba conf file appropriate to the POS installed.

=item B<--samba-set-passdb>

Configure samba to use a "passdb backend" of "smbpasswd".

=item B<--samba-rebuild-passdb>

Rebuild the samba "smbpasswd" file.

=item B<--init-console-res>

Initialize the console resolution by editing the GRUB config file.
All kernel lines in the config file that have not already been
appropriately modified will be changed.

=item B<--enable-boot-msgs>

Enable verbose boot messages by editing the GRUB config file.
All kernel lines in the config file that have not already been
appropriately modified will be changed.

=item B<--disable-kms>

Disable kernel (video) mode setting by editing the GRUB config file.
All kernel lines in the config file that have not already been
appropriately modified will be changed.

=item B<--cupsconf>

Edit the F</etc/cups/cupsd.conf> file.

=back


=head1 DESCRIPTION

This I<updateos.pl> script provides many essential methods used to setup and
configure a Red Hat Linux system for use as a Teleflora POS server.

=head2 COMMAND LINE OPTIONS

The C<"--java"> command line option downloads and installs
the Oracle Java SE JRE package.
The Java package file
for RHEL 32-bit servers is downloaded from
C<"http://rtihardware.homelinux.com/ks/jre-latest-linux-i586-rpm.bin">, and
for RHEL 64-bit servers is downloaded from
C<"http://rtihardware.homelinux.com/ks/jre-latest-linux-x64-rpm.bin">.

The C<"--java-version=s"> command line option may be used to specify a
specific version of the Java SE JRE package.
For example, to install the version of the Java JRE from package file
C<"jre-6u31-linux-x64-rpm.bin"> on a 64-bit server,
specify C<"--java-version=6u31">.
The default version is "latest".

The C<"--rti14"> command line option can be specified to make system
configuration changes appropriate for RTI version 14.
These changes include the following:

=over 4

=item IP addr

Check the method of booting in C</etc/sysconfig/network-scripts/ifcfg.eth0> and
if it's "DHCP", log an error and exit.
Actually, the device name may be specified with the C<--ifname> option and
defaults to "eth0" if C<--ifname> is not specified.

=item File System Remounting

The C</teleflora> file system is remounted as the C</usr2> file system.

=item Package installation

The Oracle Java package is installed.
If running on a RHEL4 system, the C<apache> package is installed.
If running on a RHEL5 system, the C<apache>, C<fetchmail>, C<uucp>, and C<ksh>
packages are installed.
If running on a RHEL6 system, the C<apache>, C<fetchmail>, and C<ksh>
packages are installed except that

=item Samba Configuration

If the Samba configuration has not already been modified,
generate the Samba configuration appropriate for the installed POS,
either "RTI" or "Daisy".

=back

The C<--syslog-mark> command line option can be specified to
configure I<syslog> on RHEL5 systems and I<rsyslog> on RHEL6 systems
to write a "mark" message to the syslog log file at
the specified period.

For I<rsyslog> on RHEL6:

The following 2 lines must be added to the beginning of
the I</etc/syslog.conf> conf file:

$ModLoad immark.so
$MarkMessagePeriod 1200

The "$ModLoad" line is a directive to load the input module "immark.so"
which provides the mark message feature.
The "$MarkMessagePeriod" line is a directive to specify the number of
seconds between mark messages.
This directive is only available after the
"immark" input module has been loaded.
Specifying 0 is possible and disables mark messages.
In that case, however, it is more efficient to NOT load
the "immark" input module.
In general, the last directive to appear in file wins.

For I<syslog> on RHEL5:

In the I</etc/sysconfig/syslog> conf file,
the "SYSLOGD" variable must have the "-m" option
changed to have a non-zero value if it appears.
If the "-m" option does not appear, then the default value for
the mark message period is 20 minutes.
If the value specified with the "-m" option is 0, then the
mark message is not written.


The C<--kernel-msg-console> command line option can be specified to
configure I<syslog> on RHEL5 systems and I<rsyslog> on RHEL6 systems
to direct kernel messages to specified tty device.
The syslog system service will be restarted after a succeful change
to the config file.

The C<--samba-gen-conf> command line option generates an entirely
new Samba conf file appropriate to POS installed,
either "RTI" or "Daisy".
The "RTI" and "Daisy" version of the Samba conf file are
considerably different.
If a new Samba conf file is generated, the Samba system service "smb"
is restarted.

The C<--samba-set-passdb> command line option can be used to
configure samba to use a "passdb backend" of "smbpasswd".
It does this by editing the existing Samba conf file, adding
the parameter "passdb backend = smbpasswd" to the "[global]"
section of the Samba conf file.
If the Samba conf file is already so configured,
then no change is done.
If the Samba conf file is modified, the Samba system service "smb"
is restarted.

The C<--samba-rebuild-passdb> command line option can be used to
rebuild the samba "smbpasswd" file.
The only field updated in "smbpasswd" file is the "UID" field:
if the value in the "UID" field of the "smbpasswd" file
does not match the "UID" field in the "/etc/passwd" file
for the username in the "username" field of the "smbpasswd" file,
then the UID from the "/etc/passwd" file is substituted for the
value in the "smbpasswd" file.

The C<--disable-kms> command line option is used to disable
kernel (video) mode setting.
For Daisy customers that use the console of the Daisy server as a workstation,
disabling KMS is required or Daisy application screens do not appear
correctly on the virtual consoles.
The code for this option is also executed when the
C<--baremetal> option is specified.

Specifying the C<--cupsconf> command line option will cause
F<udpateos.pl> to edit the values of several variables in the
F</etc/cups/cupsd.conf> CUPS config file.
The code for this option is also executed when the
C<--baremetal> option is specified.
The following variables are set to the specified values:

=over

=item o

MaxJobs is set to 10000

=item o

PreserveJobHistory is set to "No"

=item o

PreserveJobFiles is set to "No"

=item o

If the platform is "RHEL5", Timeout is set to 0.
If the platform is "RHEL6", Timeout is set to 300.

=item o

ErrorPolicy is set to "retry-job"

=back

=head1 FILES

=over 4

=item B</boot/grub/grub.conf>

The GRUB config file.

=item B</etc/sysconfig/syslog>

The configuration file for the syslog system service that
needs to be edited for configuring the heartbeat.
This is for RH72, FC3, FC5, RHEL4, RHEL5 only.

=item B</etc/rsyslog.conf>

For RHEL6 systems, the configuration file for the syslog system service that
needs to be edited for configuring the heartbeat.

=item B</etc/samba/smb.conf>

The Samba configuration file.

=item B</etc/samba/smbpasswd>

The Samba password file when the "passdb backend = smbpasswd" is
specified in the Samba conf file.

=item B</var/lib/samba/private/smbpasswd>

For RHEL6 systems,
the Samba password file when the "passdb backend = smbpasswd" is
specified in the Samba conf file.

=item B<jre-latest-linux-i586-rpm.bin>

The latest version of the Java SE JRE package file
for RHEL 32-bit servers.

=item B<jre-latest-linux-x64-rpm.bin>

The latest version of the Java SE JRE package file
for RHEL 64-bit servers.

=item B</etc/cups/cupsd.conf>

The values of several variables are set.

=back


=head1 DIAGNOSTICS

=over 4

=item Exit status 0 ($EXIT_OK)

Successful completion.

=item Exit status 1 ($EXIT_COMMAND_LINE)

In general, there was an issue with the syntax of the command line.

=item Exit status 2 ($EXIT_MUST_BE_ROOT)

For all command line options other than "--version" and "--help",
the user must be root or running under "sudo".

=item Exit status 3 ($EXIT_SAMBA_CONF)

During the execution of "--samba-set-passdb" or
"--samba-rebuild-passdb", either the Samba conf file is missing, or
can't be modified.

=item Exit status 4 ($EXIT_GRUB_CONF)

An unexpected error occurred during editing the GRUB config file.
The original GRUB config file will be left unchanged.

=item Exit status 5 ($EXIT_ARCH)

The machine architecture of the system is unsupported (should not happen).

=item Exit status 10 ($EXIT_JAVA_VERSION)

The name of the Java JRE package to be downloaded from B<"rtihardware.homelinux.com">
could not be determined.

=item Exit status 11 ($EXIT_JAVA_DOWNLOAD)

The Java JRE package
from B<"rtihardware.homelinux.com"> could not be downloaded.

=item Exit status 12 ($EXIT_JAVA_INSTALL)

The RPM from the download of the Java JRE package
from B<rtihardware.homelinux.com> could not be installed.

=item Exit Status 13 ($EXIT_RTI14)

The "--rti14" command line option was run and the system was not
configured with a static IP address.

=item Exit Status 15 ($EXIT_READAHEAD)

The "--uninstall-readahead" command line option was specified
on a RHEL6 system, and the "readahead" RPM could not be removed.

=item Exit status 21 ($EXIT_WRONG_PLATFORM)

The "--ospatches" option was run on an unsupported platform.

=item Exit status 22  ($EXIT_RHWS_CONVERT)

The "--ospatches" option was run on a Red Hat Workstation 5 system and
the conversion from "workstation" to "server" failed.

=item Exit status 23 ($EXIT_UP2DATE)

The "--ospatches" option was run on a RHEL 4 server system and
the "up2date" process failed.

=item Exit status 24 ($EXIT_YUM_UPDATE)

The "--ospatches" option was run on a RHEL 5 or 6 server system and
the "yum" process failed.

=item Exit status 25 ($EXIT_DIGI_DRIVERS)

The "--ospatches" option was run and the installation of the Digi Drivers
failed.

=item Exit status 26 ($EXIT_INITSCRIPTS = 26)

The "--ospatches" option was run and the "fixup" required for
RHEL6 Daisy systems failed.  This "fixup" is only required when
there is the installation of any updated "initscripts" pacakges;
however, the "fixup" is run anytime the "--ospatches" option is
run and the "yum" command which runs returns successful exit status.
(the "fixup" merely consists of removing two files if they exist:
C</etc/init/start-ttys.conf> and C</etc/init/tty.conf>)

=item Exit status 27 ($EXIT_RHN_NOT_REGISTERED = 27)

The system is not registered with the Red Hat Network, and
thus patches can not be dowloaded from Red Hat.

=item Exit status 30 ($EXIT_HOSTNAME_CHANGE = 30)

The attempt to change the hostname of the system failed.

=back


=head1 SEE ALSO

/var/log/messages


=cut
