#! /usr/bin/perl 
#
# $Revision: 1.6 $
# Copyright 2009 Teleflora
#
# Report info about the Teleflora Daisy POS installation.
#

use strict;
use warnings;
use POSIX;
use Getopt::Long;
use English;


#=====================================#
# +---------------------------------+ #
# | Global variables                | #
# +---------------------------------+ #
#=====================================#
my $HARDWARE = "unknown";
my $DEFAULT_DSYDIR = '/d/daisy';
my $DSYDIR = "";
my $VERSION = 0;
my $HELP = 0;


#=====================================#
# +---------------------------------+ #
# | Utility Functions               | #
# +---------------------------------+ #
#=====================================#
sub usage
{
	print "Usage:\n";
	print "$0" . ' $Revision: 1.6 $ ' . "\n";
	print "$0 --help         # output this help message and exit\n";
	print "$0 --version      # output version number and exit\n";
	print "$0                # report info about Daisy installation\n";
	print "\n";

	return "";
}

#
# PCI 6.3
#	Develop software appliations in accordance with PCI DSS and
#	based on industry best practices, and incorporate information
#	security throughout the software development life cycle.
#
#	PCI 6.3.1.1
#	Validation of all input.
#
# Examples of how potentially dangerous input is converted:
#
# "some string; `cat /etc/passwd`" -> "some string cat /etc/passwd"
# "`echo $ENV; $(sudo ls)`" -> "echo ENV sudo ls"
#
sub validate_input
{
        my $var = $_[0];
        my $temp = "";

        if(! $var) {
                return "";
        }

        $temp = $var;

        # "`bad command`" -> "bad command"
        $temp =~ s/\`//g;

        # "$(bad command)" -> "bad command"
        $temp =~ s/(\$\()(.*.)(\))/$2/g;


        # "stuff ; bad command" -> "stuff bad command"
        $temp =~ s/\;//g;

        # "stuff && bad command" -> "stuff bad command"
        $temp =~ s/\&//g;

        # "stuff | bad command" -> "stuff bad command"
        $temp =~ s/\|//g;

        # "stuff > bad command" -> "stuff bad command"
        $temp =~ s/\>//g;

        # "stuff < bad command" -> "stuff bad command"
        $temp =~ s/\<//g;

        # Filter non printables
        $temp =~ s/[[:cntrl:]]//g;


        return($temp);
}

#=====================================#
# +---------------------------------+ #
# | Application Specific Functions  | #
# +---------------------------------+ #
#=====================================#

sub report_dove_info
{
	my $dsydir = $_[0];
	my $trcv_data_file = "$dsydir/trcvdat.pos";
	my $dove_control_data_file = "$dsydir/dovectrl.pos";

	unless (-f "$trcv_data_file") {
		return;
	}

	#
	# get just the mtime of the data file
	#
	my $mtime = (stat($trcv_data_file))[9];

	my $lastmonth = time() - 30*24*60*60; # 30 days per month, 24 hrs/day 60 mins/hr 60 secs/min
	if ($mtime < $lastmonth) {
		print("Dove has received no data for a month");
		return;
        }

	#
	# If there is an active Dove, then what is the shop ID
	# entered for Primary ID by the florist.
	#
	unless (-f "$dove_control_data_file") {
		return;
	}

	my ($buffer) = "";

	unless (open(DF, $dove_control_data_file)) {
		print("Unable to determine Teleflora Dove ID\n");
		return;
	}

	sysread(DF, $buffer, 50);

	close(DF);

	printf("Teleflora Dove ID is %s\n", substr($buffer, 30, 9));

}

sub report_directory_info
{
	my $dsydir = $_[0];
	my $edir_data_file = "$dsydir/control.tel";

	unless (-f "$edir_data_file") {
		return;
	}

	#
	# just get the mtime of the data file
	#
	my $mtime = (stat($edir_data_file))[9];

	# Is the Teleflora directory current in last year

	my $lastyear = time() - 365*24*60*60;
	if ($mtime < $lastyear) {
		print("Teleflora directory not updated for a year");
		return;
        }

	unless(open(DF, $edir_data_file)) {
		print("Unable to determine Teleflora directory quarter");
		return;
	}

	my $firstline = <DF>;

	# First line of control.tel must start with Teleflora or something is very wrong
	unless (substr($firstline, 0, 9) eq "Teleflora") {
		print("Unable to determine Teleflora directory quarter");
		return;
        }

	my $quarter = "";

	if (index($firstline,"Feb",9) > 0) {
		$quarter = "FMA";
		}
	elsif (index($firstline,"May",9) > 0) {
		$quarter = "MJJ";
		}
	elsif (index($firstline,"Aug",9) > 0) {
		$quarter = "ASO";
		}
	elsif (index($firstline,"Nov",9) > 0) {
		$quarter = "NDJ";
        } else {
		$quarter = "???";
        }

	if ($firstline =~ m/(20\d\d)/ ) {
		print "Teleflora quarterly directory release is: ", $1, $quarter, "\n";
        }
}


sub report_daisy_version_nr
{
	my $dsydir = $_[0];
	my $identify_cmd = "$dsydir/bin/identify_daisy_distro.pl";

	unless (-x $identify_cmd) {
		print("Can't identify Daisy version - try running \"pos --version\".\n");
	}

	my $line = qx($identify_cmd $dsydir);
	chomp $line;
	my ($product_name, $product_os, $product_country, $product_version, $product_dir) =
		split(/[[:space:]]/, $line);
	print("Daisy version: $product_version\n");
}


sub identify_hardware
{

	open(PIPE, "/usr/sbin/dmidecode |");
	while(<PIPE>) {
		if(/PowerEdge T300/) {
			$HARDWARE = "t300";
		} elsif (/PowerEdge T310/) {
			$HARDWARE = "t310";
		} elsif (/PowerEdge T410/) {
			$HARDWARE = "t410";
		} elsif (/PowerEdge 1800/) {
			$HARDWARE = "1800";
		} elsif (/PowerEdge 420/) {
			$HARDWARE = "420";
		} elsif (/PowerEdge R910/) {
			$HARDWARE = "r910";
		} elsif (/PowerEdge 2950/) {
			$HARDWARE = "2950";
		} elsif (/Precision 380/) {
			$HARDWARE = "Precision380";
		} elsif (/Precision 390/) {
			$HARDWARE = "Precision390";
		} elsif (/Precision T3400/) {
			$HARDWARE = "t3400";
		} elsif (/Precision T3500/) {
			$HARDWARE = "t3500";
		} else {
			$HARDWARE = "unknown";
		}
	}
	close(PIPE);
}

#=====================================#
# +---------------------------------+ #
# | Main                            | #
# +---------------------------------+ #
#=====================================#

GetOptions (
	"version" => \$VERSION,
	"help" => \$HELP,
);


# --version
if($VERSION != 0) {
	print("OSTools Version: 1.14.0\n");
	print '$Revision: 1.6 $' . "\n";
	exit 0;
}


# --help
if($HELP != 0) {
	usage();
	exit 0;
}

#
# If there is more than one command line argument, it's an error.
# If there is one command line argument, use it as the path to the
# daisy database directory.  If there are no command line args,
# then use the default value.
#
if (@ARGV > 1) {
	usage();
	exit(1);
} elsif (@ARGV == 1) {
	$DSYDIR = $ARGV[0];
} else {
	$DSYDIR = $DEFAULT_DSYDIR;
}

#
# Validate user input.
#
$DSYDIR = validate_input($DSYDIR);
unless ($DSYDIR) {
	$DSYDIR = $DEFAULT_DSYDIR;
}

# make sure it is an absolute path
unless (substr($DSYDIR, 0, 1) eq '/') {
	print("\"$DSYDIR\" must be an absolute path.\n");
	usage();
	exit(1);
}

# make sure it's a daisy database directory
unless (-f "$DSYDIR/control.dsy") {
	print("\"$DSYDIR\" is not a daisy database directory.\n");
	usage();
	exit(1);
}

#
# Step 1 - is there an active Dove system in this directory
#
report_dove_info($DSYDIR);
report_directory_info($DSYDIR);
report_daisy_version_nr($DSYDIR);

exit(0);


__END__

=pod

=head1 NAME

tfinfo.pl - report information about the Teleflora Daisy POS


=head1 VERSION

This documenation refers to version: $Revision: 1.6 $


=head1 USAGE

tfinfo.pl

tfinfo.pl B<--version>

tfinfo.pl B<--help>


=head1 OPTIONS

=over 4

=item B<--version>

Output the version number of the script and exit.

=item B<--help>

Output a short help message and exit.

=back


=head1 DESCRIPTION

The I<tfinfo.pl> script gathers information about a Daisy system,
such as Dove activity, edir version number, and Daisy version number, and
writes a short summary to stdout.

The script C<identify_daisy_distro.pl> is used to get the Daisy version number.


=head1 FILES

=over 4

=item B</d/daisy/control.dsy>

The Daisy control file.

=item B</d/daisy/control.tel>

The Daisy edir control file.

=item B</d/daisy/dovectrl.pos>

The Daisy Dove control file.

=back


=head1 DIAGNOSTICS

=over 4

=item Exit status 0 ($EXIT_OK)

Successful completion.

=item Exit status 1 ($EXIT_COMMAND_LINE)

In general, there was an issue with the syntax of the command line.

=back


=head1 SEE ALSO

identify_daisy_distro.pl


=cut
