#!/usr/bin/perl
#
# $Revision: 1.50 $
# Copyright 2009-2011 Teleflora
# 
# Setup a "parallel" SSHd service called "tfremote".
# PA-DSS 11.2
# PCI-DSS 8.2, 8.3
#

use strict;
use warnings;
use Digest::MD5;
use Getopt::Long;
use POSIX;
use English;
use File::Basename;

my $CVS_REVISION = '$Revision: 1.50 $';
my $TIMESTAMP = strftime("%Y%m%d%H%M%S", localtime());
my $PROGNAME = basename($0);

my $OS = "";
my $VERSION = 0;
my $HELP = 0;
my $INSTALL = 0;
my $START = 0;
my $STOP = 0;
my $STATUS = 0;
my $CONNECT = "";

GetOptions(
	"version" => \$VERSION,
	"help" => \$HELP,
	"install" => \$INSTALL,
	"start" => \$START,
	"stop" => \$STOP,
	"status" => \$STATUS,
	"connect=s" => \$CONNECT,
);



# --help
if($HELP != 0) {
	usage();
	exit(0);
}

# --version
if($VERSION != 0) {
	print("OSTools Version: 1.14.0\n");
	print("$CVS_REVISION\n");
	exit(0);
}

# --status
if($STATUS != 0) {
	exit(tfremote_status());
}


# --connect some.host.com
# --connect=some.host.com
if("$CONNECT" ne "") {
	exit(connect_to_host($CONNECT));
}


############
# Root is needed to execute commands below.
############
if($< != 0) {
	usage();
	print("Error: Must run this script as 'sudo' or root.\n");
	exit(1);
}


# --start
if($START != 0) {
	exit(start_tfremote());
}

# --stop
if($STOP != 0) {
	exit(stop_tfremote());
}

# --install
if($INSTALL != 0) {
	exit(install_tfremote());
}




usage();

exit(0);
#####################################################################
#####################################################################
#####################################################################


sub usage
{
	print("Usage:\n");
	print("$0 --help\n");
	print("$0 --version\n");
	print("$0 --start\n");
	print("$0 --stop\n");
	print("$0 --status\n");
	print("$0 --install\n");
	print("$0 --connect some.host.com\n");
	print("$0 --connect user\@sftp.teleflora.com\n");
	print("\n");
	print("TFRemote is a system service which only allows incoming connections\n");
	print("authenticated via ssh public key encryption.  Since the service is\n");
	print("actually a second instance of OpenSSH, all configuration and\n");
	print("usage rules which apply to SSH, apply here as well.\n");
	print("\n");
	print("This script is capable of setting up the 'tfremote' service on a\n");
	print("Red Hat Enterprise Linux system, as well as starting and stopping\n");
	print("the service.\n");
	print("\n");
	return(0);
}


#
# Get useful information such as groups and home directory
# about a given user.
sub get_userinfo
{
	my $username = $_[0];
	my $groupname = "";
	my $users = "";
	my @ent = ();
	my @groups = ();
	my %hash;
	my $found = 0;

	if($username eq "") {
		return %hash;
	}

	# Get user related info.
	setpwent();
	@ent = getpwent();
	while(@ent) {
		if( "$ent[0]" eq "$username") {
			$found = 1;
			$hash{"username"} = $username;
			$hash{"homedir"} = "$ent[7]";
			$hash{"shell"} = "$ent[8]";
			last;
		}
		@ent = getpwent();
	}
	endpwent();
 

	# We did  not find this user.
	if($found == 0) {
		return %hash;
	}


	# Get user's associated groups.
	# ($name,$passwd,$gid,$members) = getgr*
	$hash{"groups"} = ();
	@ent = getgrent();
	while(@ent) {
		$groupname = $ent[0];
		$users = $ent[3];
		if($users =~ /([[:space:]]*)($username)([[:space:]]{1}|$)/) {
			push @groups, $groupname;
		}
		@ent = getgrent();
	}
	endgrent();
	$hash{'groups'} = [ @groups ];
 

	return(%hash);
}


sub start_tfremote
{
	exec("/sbin/service tfremote start");
}


sub stop_tfremote
{
	exec("/sbin/service tfremote stop");
}


sub tfremote_status
{
	exec("/sbin/service tfremote status");
}

sub add_service
{
    my $service_name = $_[0];
    my $service_levels = $_[1];

    system("/sbin/chkconfig --add $service_name");
    system("/sbin/chkconfig --level $service_levels $service_name on");
}

sub start_service
{
    my $service_name = $_[0];

    system("/sbin/service $service_name start");
}


#
# Create the system service.
# This is basically a clone of "sshd" which runs concurrent to sshd,
# uses the same config files, but is just renamed.  The reason for
# a separate, concurrent process, is to enforce on a "system" level,
# the use of two-factor authentication.
#
sub install_tfremote
{
	my $prog = "";
	my $file = "";
	my $string = "";


	# Verify our "source files" exist.
	foreach $file ("/etc/init.d/sshd", "/etc/ssh/sshd_config", "/usr/sbin/sshd") {
		unless (-f "$file") {
			print("Error: \"$file\". $!\n");
			return(1);
		}
	}

	foreach $file ("/etc/sysconfig", "/etc/ssh") {
		unless (-d "$file") {
			print("Error: \"$file\" $!\n");
			return(2);
		}
	}

	# /etc/init.d
	open(OLD, "< /etc/init.d/sshd");
	open(NEW, "> /etc/init.d/tfremote");
	while(<OLD>) {
	    $string = $_;
	    $string =~ s/ssh_host_key/tfremote_host_key/g;
	    $string =~ s/ssh_host_rsa_key/tfremote_host_rsa_key/g;
	    $string =~ s/ssh_host_dsa_key/tfremote_host_dsa_key/g;

	    $string =~ s/ssh_random_seed/tfremote_random_seed/g;
	    $string =~ s/sshd/tfremote/g;
	    $string =~ s/prog=\"([[:print:]]+)\"/prog=\"tfremote\"/g;
	    $string =~ s/description: .+$/description: Teleflora POS Remote Access/;

	    print(NEW $string);
	}
	close(OLD);
	close(NEW);
	system("chmod --reference=/etc/init.d/sshd /etc/init.d/tfremote");
	system("chown --reference=/etc/init.d/sshd /etc/init.d/tfremote");


	system("mkdir -p /var/empty/tfremote");
	system("chmod --reference=/var/empty/sshd /var/empty/tfremote");
	system("chown --reference=/var/empty/sshd /var/empty/tfremote");


	# Make the service start at bootup.
	# Must use a hard link here (not symlink), as, RHEL init.d/functions will
	# wind up killing all "sshd" processes if we use a symlink.
	# Redhat updates should work fine in the event that we use hard links.
	unlink("/usr/sbin/tfremote");
	#
	# FIXME: using hard links is an issue for updates.
	#
	system("ln /usr/sbin/sshd /usr/sbin/tfremote");


	# tfremote_config
	open(OLD, "< /etc/ssh/sshd_config");
	open(NEW, ">  /etc/ssh/tfremote_config");
	while(<OLD>)
	{
		next if(/Port /i);
		next if(/ListenAddress /i);
		next if(/Protocol /i);
		next if(/HostKey /i);
		next if(/AcceptEnv /i);
		next if(/SyslogFacility /i);
		next if(/LogLevel /i);
		next if(/LoginGraceTime /i);
		next if(/MaxAuthTries /i);
		next if(/MaxStartups /i);
		next if(/PidFile /i);
		next if(/PermitRootLogin /i);
		next if(/RSAAuthentication /i);
		next if(/PubkeyAuthentication /i);
		next if(/RhostsRSAAutthenitcation /i);
		next if(/HostbasedAuthentication /i);
		next if(/IgnoreUserKnownHosts /i);
		next if(/IgnoreRhosts /i);
		next if(/PermitEmptyPasswords /i);
		next if(/PasswordAuthentication /i);
		next if(/KerberosAuthentication /i);
		next if(/UsePrivilegeSeparation /i);
		next if(/GSSAPIAuthentication /i);
		next if(/UsePAM /i);
		next if(/Banner /i);
		next if(/ClientAliveInterval /i);
		next if(/ClientAliveCountMax /i);

		print(NEW);
	}
	close(OLD);


	print(NEW "\n");
	print(NEW "\n");
	print(NEW "#\n"); 
	print(NEW "# PA-DSS Compliant Settings\n");
	print(NEW "# Setup by tfremote " . '$Revision: 1.50 $' . "\n");
	print(NEW "#\n"); 
	print(NEW "Port 15022\n");
	print(NEW "ListenAddress 0.0.0.0\n");
	print(NEW "Protocol 2\n");
	print(NEW "HostKey /etc/ssh/tfremote_host_rsa_key\n");
	print(NEW "HostKey /etc/ssh/tfremote_host_dsa_key\n");
	print(NEW "AcceptEnv BBTERM\n");
	print(NEW "SyslogFacility AUTH\n");
	print(NEW "LogLevel VERBOSE\n");
	print(NEW "LoginGraceTime 30\n");
	print(NEW "MaxAuthTries 1\n");
	print(NEW "MaxStartups 3:30:10\n");
	print(NEW "PidFile /var/run/tfremote.pid\n");
	print(NEW "PermitRootLogin no\n");
	print(NEW "RSAAuthentication no\n");
	print(NEW "PubkeyAuthentication yes\n");
	print(NEW "RhostsRSAAuthentication no\n");
	print(NEW "HostBasedAuthentication no\n");
	print(NEW "IgnoreUserKnownHosts yes\n");
	print(NEW "IgnoreRhosts yes\n");
	print(NEW "PermitEmptyPasswords no\n");
	print(NEW "PasswordAuthentication no\n");
	print(NEW "KerberosAuthentication no\n");
	print(NEW "UsePrivilegeSeparation yes\n");
	print(NEW "GSSAPIAuthentication no\n");
	print(NEW "UsePAM no\n");
	print(NEW "AllowTcpForwarding yes\n");
	print(NEW "X11Forwarding no\n");
	print(NEW "Banner /etc/motd\n");
	print(NEW "ClientAliveInterval 300\n");
	print(NEW "ClientAliveCountMax 3\n");
	print(NEW "\n");
	print(NEW "\n");
	close(NEW);

	unless (-f "/etc/ssh/tfremote_config") {
		print("Error: /etc/ssh/tfremote_config  $!\n");
		return(5);
	}


	# Here is where we tell 'sshd' to read our PA-DSS compliant settings.
	open(NEW, "> /etc/sysconfig/tfremote");
	print(NEW "OPTIONS=\"-f /etc/ssh/tfremote_config\"\n");
	close(NEW);

	unless (-f "/etc/sysconfig/tfremote") {
		print("Error: /etc/sysconfig/tfremote $!\n");
		return(6);
	}

	# Make sure the service starts with each reboot.
	add_service("tfremote", "35");

	# start the service
	start_service("tfremote");
}


#
# Connect to a tfremote server.
#
# Doubles as a "sftp client" to help us connect to sftp.teleflora.com.
# This script is used mainly on our customer service servers to connect
# to customer machines; setting up appropriate security parameters,
# as well as port forwarding rules.
#
sub connect_to_host
{
	my $hostname = $_[0];
	my %userinfo = ();
	my $localuser = getpwuid($>);
	my $command = "";
	my $portnum = int(10000 + int($UID));
	my $remoteuser = "";
	my $id_dsa = "";
	my $thisfile = "";
	my $fingerprint = "";
	my @array = ();

	if (!defined($hostname)) {
	    return(1);
	}

	# Who is the current user?
	if ($localuser eq "") {
		log_error("Could not determine name of effective user.");
		return (1);
	}

	# We want to sftp out.
	# In some cases, we have had issues where the tfsupport user account
	# has an 'id_dsa' or 'id_rsa' in their .ssh directory.  With default "sftp",
	# these private keys are presented to sftp.teleflora.com, which denies
	# the login and fails to allow a failover to password.
	# Thus, a sftp command is used that specifies no pubkey authentication and
	# only uses password authentication.
	#
	if ($hostname =~ /sftp\.teleflora\.com/i) {
	    my $sftp_cmd = '/usr/sbin/sftp';
	    my $sftp_opts = '-o PubKeyAuthentication=no';
	    my $sftp_site = 'sftp.teleflora.com';
	    my $sftp_url = "";

	    if($hostname =~ /(\S+)(\@)(sftp)/) {
		$sftp_url = $hostname;
	    }
	    if (-d '/d/daisy') {
		$sftp_url = "daisydev\@$sftp_site";
	    }
	    if (-d '/usr2/bbx') {
		$sftp_url = "rtidev\@$sftp_site";
	    }

	    if ($sftp_url eq "") {
		log_error("Could not determine URL of sftp site.");
		return (1);
	    }

	    system("$sftp_cmd $sftp_opts $sftp_url");
	    return($?);
	}


	# Where is our home directory?
	%userinfo = get_userinfo($localuser);
	foreach $thisfile ("tfremote-id_dsa", "tfsupport-id_dsa", "id_dsa") {
	    if(-f "$userinfo{'homedir'}/.ssh/$thisfile") {
		$id_dsa = "$userinfo{'homedir'}/.ssh/$thisfile";
		open(PIPE, "/usr/bin/md5sum $id_dsa |");
		while(<PIPE>) {
		    chomp;
		    @array = split(/\s+/);
		    if($#array >= 0) {
			$fingerprint .= $array[0];
		    }
		}
		close(PIPE);
		last;
	    }
	}
	if($id_dsa =~ /tfsupport-id_dsa/) {
	    $remoteuser = "tfsupport";
	} else {
	    $remoteuser = $localuser;
	}

	if ($id_dsa eq "") {
	    log_error("Could not find a DSA SSH Private Key File for $localuser.");
	    return(2);
	}

	unless (-f $id_dsa) {
	    log_error("Could not find DSA SSH Private Key File ($id_dsa) for $localuser.");
	    return(3);
	}


	# Log when we start and end our session, as well as who we are.
	system("/usr/bin/logger \"$localuser:$fingerprint:$$ SSH to $remoteuser\@$hostname\"");

	$command = "ssh ";
	$command .= " -t";
	$command .= " -o PubKeyAuthentication=yes";
	$command .= " -o PasswordAuthentication=no";
	$command .= " -o GSSAPIAuthentication=no";
	$command .= " -o StrictHostKeyChecking=no";
	$command .= " -o LogLevel=VERBOSE";
	$command .= " -o DynamicForward=$portnum";
	$command .= " -i $id_dsa";
	$command .= " -l $remoteuser";
	$command .= " $hostname";
	$command .= " \"/bin/bash --login\"";


	print("\n");
	print("\n");
	print("Your SOCKS Port is: $portnum\n");
	print("\n");
	print("Suggested Putty configuration:\n");
	print("Connection->SSH->Tunnels->Source port=10000, Destination=localhost:$portnum\n");
	print("\n");
	print("Suggested Firefox configuration:\n");
	print("Manual Proxy Configuration->Host or IP Address=localhost Port=10000\n");
	print("\n");
	print("\n");

	system("$command");
	if ($? == 0) {
	    system("/usr/bin/logger \"$localuser:$fingerprint:$$ LOGOUT from $remoteuser\@$hostname status SUCCESS\"");
	}
	else {
	    system("/usr/bin/logger \"$localuser:$fingerprint:$$ LOGOUT from $remoteuser\@$hostname status FAIL ($?)\"");
	}

	return(0);
}


sub log_error
{
    my $emsg = $_[0];

    system("/usr/bin/logger $emsg");
    print("Error: $emsg\n");
}


__END__

=pod

=head1 NAME

tfremote.pl - script for remote access to a Teleflora POS


=head1 VERSION

This documenation refers to version: $Revision: 1.50 $


=head1 USAGE

tfremote.pl

tfremote.pl B<--version>

tfremote.pl B<--help>

tfremote.pl B<--install>

tfremote.pl B<--start>

tfremote.pl B<--stop>

tfremote.pl B<--status>

tfremote.pl B<--connect=s>


=head1 OPTIONS

=over 4

=item B<--version>

Output the version number of the script and exit.

=item B<--help>

Output a short help message and exit.

=back


=head1 DESCRIPTION

The I<tfremote.pl> script sets up a "parallel" SSHd service called "tfremote".
The only authentication method allowed is "public key".


=head1 FILES

=over 4

=item B</etc/init.d/tfremote>

=item B</etc/ssh/tfremote_config>

=item B</etc/sysconfig/tfremote>

=item B</usr/sbin/tfremote>


=back


=head1 DIAGNOSTICS

=over 4

=item Exit status 0

Successful completion.

=item Exit status 1

For all command line options other than "--version",
"--help",
"--status", or
"--connect",
the user must be root or running under "sudo".

=back


=head1 SEE ALSO

sshd(8)


=cut
