#! /usr/bin/perl 
#
# $Revision: 1.1 $
# Copyright 2010 Teleflora
#
# Teleflora patch file downloader... get an edir patch from
# the Teleflora "blob" server for either RTI or Daisy and
# put it where it will be ready to install.
#

use strict;
use warnings;
use POSIX;
use Getopt::Long;
use English;
use File::Basename;

my $CVS_REVISION = '$Revision: 1.1 $';
my $TIMESTAMP = strftime("%Y%m%d%H%M%S", localtime());
my $PROGNAME = basename($0);

#
# Alternate os and root for testing.
#
my $ALTROOT = "";
my $ALTOS = "";

#=====================================#
# +---------------------------------+ #
# | Global variables                | #
# +---------------------------------+ #
#=====================================#
my $VERSION = 0;
my $HELP = 0;
my $RTI = 0;
my $DAISY = 0;

my $RTIDIR = "$ALTROOT/usr2/bbx";
my $DAISYDIR = "$ALTROOT/d/daisy";
my $TOOLDIR = "";

my $RTI_TEL_PATCH_FILE = "/rti-edir-tel-latest.patch";
my $RTI_CMB_PATCH_FILE = "/rti-edir-cmb-latest.patch";
my $DAISY_EDIR_PATCH_FILE = "/daisy-edir-latest.patch";

my $BLOB_SERVER_URL = "http://tposlinux.blob.core.windows.net";
my $BLOB_SERVER_RTI_URL = $BLOB_SERVER_URL . "/rti-edir";
my $BLOB_SERVER_DAISY_URL = $BLOB_SERVER_URL . "/daisy-edir";

my $RTI_FLORDIR = "/usr2/FLORDIR";
my $DAISY_FLORDIR = "/d/FLORDIR";

my $LOGFILE = 'RTI-Patches.log';


#=====================================#
# +---------------------------------+ #
# | Exit status values              | #
# +---------------------------------+ #
#=====================================#
my $EXIT_OK = 0;
my $EXIT_COMMAND_LINE = 1;
my $EXIT_MUST_BE_ROOT = 2;
my $EXIT_POS_NOT_PRESENT = 4;
my $EXIT_GET_PATCH = 5;


#
# The command line must be recorded before the GetOptions modules
# is called or any options will be removed.
#
my $COMMAND_LINE = get_command_line();

GetOptions (
	"version" => \$VERSION,
	"help" => \$HELP,
	"rti" => \$RTI,
	"daisy" => \$DAISY,
	"rti-tel-file=s" => \$RTI_TEL_PATCH_FILE,
	"rti-cmb-file=s" => \$RTI_CMB_PATCH_FILE,
	"daisy-edir-file=s" => \$DAISY_EDIR_PATCH_FILE,
) || die "command line options error... exiting";


# --version
if ($VERSION != 0) {
	print "$CVS_REVISION\n";
	exit($EXIT_OK);
}


# --help
if ($HELP != 0) {
	usage();
	exit($EXIT_OK);
}


#=====================================#
# +---------------------------------+ #
# | Main                            | #
# +---------------------------------+ #
#=====================================#
sub main
{
    log_set_filename();

    loginfo("===========================================");
    loginfo("Log Entry for $PROGNAME $CVS_REVISION");
    loginfo("===========================================");
    loginfo("Invoked with command line:");
    loginfo("$COMMAND_LINE");

    # If there is a command line argument, it's an error.
    if (@ARGV) {
	usage();
	logerror("Unexpected command line argument present... exiting");
	return($EXIT_COMMAND_LINE);
    }

    if ($RTI && $DAISY) {
	logerror("Can not be RTI and Daisy at the same time... exiting");
	return($EXIT_COMMAND_LINE);
    }

    # An installed POS is a requirement.
    if ($RTI == 0 && $DAISY == 0) {
	if (-d $RTIDIR) {
	    $RTI = 1;
	}
	elsif (-d $DAISYDIR) {
	    $DAISY = 1;
	}
	else {
	    logerror("No detectable Teleflora POS installed... exiting");
	    return($EXIT_POS_NOT_PRESENT);
	}
    }

    if ($RTI) {
	$TOOLDIR = "$RTIDIR/bin";
    }
    elsif ($DAISY) {
	$TOOLDIR = "$DAISYDIR/bin";
    }

    my $exit_status = $EXIT_OK;

    if (get_patch() != 0) {
	$exit_status = $EXIT_GET_PATCH;
    }

    log_set_perms();

    return($exit_status);
}

#=====================================#
# +---------------------------------+ #
# | Utility Functions               | #
# +---------------------------------+ #
#=====================================#
sub usage
{
	print "Telelfora Patch File Downloader\n\n";
	print "Usage:\n";
	print "$PROGNAME $CVS_REVISION\n";
	print "$PROGNAME --help               # output this help message and exit\n";
	print "$PROGNAME --version            # output version number and exit\n";
	print "$PROGNAME --rti                # assume RTI system\n";
	print "$PROGNAME --daisy              # assume Daisy system\n";
	print "$PROGNAME --rti-tel-file       # name of RTI tel file on blob server\n";
	print "$PROGNAME --rti-cmb-file       # name of RTI cmb file on blob server\n";
	print "$PROGNAME --daisy-edir-file    # name of RTI edir file on blob server\n";
	print "\n";
}


sub get_command_line
{
	my $cmd_line = "";

	$cmd_line = $0;
	foreach my $i (@ARGV) {
		$cmd_line .= " ";
		if ($i =~/\s/) {
			$cmd_line .= "\"$i\"";
		} else {
			$cmd_line .= "$i";
		}
	}

	return($cmd_line);
}


# Print to screen, and log to logfile.
sub showinfo
{
	my $message = $_[0];

	print("$message\n");
	return(loginfo("$message"));
}


# Print to screen, and log to logfile.
sub showerror
{
	my $message = $_[0];

	print("Error: $message\n");
	return(logerror("$message"));
}


sub log_set_filename
{
    my $logfiledir = "";

    if (-d "$RTIDIR/log") {
	$logfiledir= "$RTIDIR/log";
    }
    elsif (-d "$DAISYDIR/log") {
	$logfiledir = "$DAISYDIR/log";
    }
    else {
	$logfiledir = "$ALTROOT/tmp";
    }

    if ($LOGFILE eq "") {
	$LOGFILE = 'RTI-Patches.log';
    }

    $LOGFILE = "$logfiledir/$LOGFILE";
}


# insurance that the log file can be accessed by non-root users
sub log_set_perms
{
    if ($ALTROOT eq "") {
	system "chown tfsupport $LOGFILE";
    }
    system "chmod 664 $LOGFILE";
}


sub loginfo
{
    my $message = $_[0];

    return(logwrite("<I> $message"));
}

sub logerror
{
    my $message = $_[0];

    return(logwrite("<E> $message"));
}

sub logdebug
{
    my $message = $_[0];

    return(logwrite("<D> $message"));
}


#
# Write message to logfile.
#
sub logwrite
{
    my $message = $_[0];

    # form line to write
    my $timestamp = strftime("%Y-%m-%d %H:%M:%S", localtime(time()));
    my $line_out = "$timestamp ($PROGNAME-$$) $message\n";

    # write logfile
    open(LOG, ">> $LOGFILE");
    print(LOG "$line_out");
    close(LOG);

    return(0);
}

#
# The following script is called:
#   /d/ostools/bin/LinuxCustominv.pl --sysinfo
#
# The sysinfo record is the last line of output.
#

sub get_sysinfo_record
{
    my $tool = "LinuxCustominv.pl --sysinfo";
    my $command = "$TOOLDIR/$tool";

    my $last_line = "";

    if (open(DF, "$command |")) {
	while(<DF>) {
	    if (eof) {
		$last_line = $_;
		chomp($last_line);
	    }
	}
	close(DF);
    }
    else {
	logerror("Could not open pipe to $command");
    }

    return($last_line);
}

#
# The sysinfo record consists of 9 COMMA separated fields that look like this:
#   00000000,DAISY,8.1.4,20120703,/d/daisy,RHEL5,ASO2012TEL,70.182.67.113,1.33.2.2
#
# The seventh field is the encoded edir release string.
# If the last 3 letters are "CMB":
#   it means they have both Teleflora and FTD files.
# If the last 3 letters are "TEL":
#   it means they have only Teleflora files.
#
# Input
#	sysinfo record
#
# Returns
#	0 if FTD files are NOT present
#	1 if FTD files are present
#

sub parse_sysinfo_record
{
    my ($sysinfo) = @_;

    chomp($sysinfo);
    my @sysinfo_record = split(',', $sysinfo);
    if (@sysinfo_record != 9) {
	logerror("Truncated sysinfo record: $sysinfo");
	return(0);
    }
 
    if ($sysinfo_record[6] =~ /.*CMB$/) {
	return(1);
    }

    return(0);
}

#
# Returns
#	"CMB" - patch type is combined FTD and Teleflora
#	"TEL" - patch type is Teleflora only
#
sub rti_patch_type
{
    my $patch_type = "TEL";

    my $sysinfo = get_sysinfo_record();
    if ($sysinfo eq "") {
	logerror("Could not get sysinfo... assuming no FTD files");
    }
    else {
	if (parse_sysinfo_record($sysinfo) == 1) {
	    $patch_type = "CMB";
	}
    }

    return($patch_type);
}


#
# Check for existence of special edir directory.  If present,
# remove all contents.  If not present, create the empty directory.
#
# Returns
#	path to flordir on success
#	empty string on error
#

sub initialize_flordir
{
    # assume RTI system but switch if Daisy
    my $flordir = $RTI_FLORDIR;
    $flordir = $DAISY_FLORDIR if ($DAISY);

    if (-d $flordir) {
	system("rm -rf $flordir/*");
    }
    else {
	system("mkdir $flordir");
    }

    unless (-d $flordir) {
	return("");
    }

    return($flordir);
}

#
# For RTI, download specified file from the RTI edir area on blob server.
# Put the downloaded file in the RTI "flordir" directory.  Same for Daisy.
#
# Returns
#	0 on success
#	1 on error
#
sub download_file
{
    my ($src_filename, $dst_dir) = @_;

    my $dst_path = "";
    my $src_url = "";
    my $rc = 0;

    if ($RTI) {
	$src_url = $BLOB_SERVER_RTI_URL . $src_filename;
	$dst_path = $dst_dir . $src_filename;
    }
    elsif ($DAISY) {
	$src_url = $BLOB_SERVER_DAISY_URL . $src_filename;
	$dst_path = $dst_dir . $src_filename;
    }
    else {
	logerror("Can't happen in download_file(): no installed POS");
	return(1);
    }

    # Curl options:
    # -f : set exit value on failure
    # -s : no progress indication
    my $curl_cmd = "curl -f -s -o $dst_path $src_url";

    system("$curl_cmd");
    if ($? != 0) {
	logerror("Could not download file from url: $src_url");
	$rc = 1;
    }

    return($rc);
}

#
# For each of RTI and Daisy, need to download a specific appplypatch.pl.
# For RTI, need to download either the combined or tel-only patch.
# For Daisy, need to download Daisy specific patch file.
#
# Returns
#	0 on success
#	1 on error
#
sub download_patch
{
    my ($dst_dir) = @_;

    my $src_filename = "";
    my $rc = 0;

    $src_filename = "/applypatch.pl";
    if (download_file($src_filename, $dst_dir) != 0) {
	logerror("Could not download patch file: $src_filename");
	$rc = 1;
    }
    else {
	if ($RTI) {
	    if (rti_patch_type() eq "CMB") {
		$src_filename = $RTI_CMB_PATCH_FILE;
	    }
	    else {
		$src_filename = $RTI_TEL_PATCH_FILE;
	    }
	    if (download_file($src_filename, $dst_dir) != 0) {
		logerror("Could not download patch file: $src_filename");
		$rc = 1;
	    }
	}
	elsif ($DAISY) {
	    $src_filename = $DAISY_EDIR_PATCH_FILE;
	    if (download_file($src_filename, $dst_dir) != 0) {
		logerror("Could not download patch file: $src_filename");
		$rc = 1;
	    }
	}
	else {
	    logerror("Can't happen in download_patch(): no installed POS");
	    $rc = 1;
	}
    }

    return($rc);
}

#=====================================#
# +---------------------------------+ #
# | Application Specific Functions  | #
# +---------------------------------+ #
#=====================================#

#
# function: get_patch
#
# Input:
#	none
#
# Returns:
#	0 if patch downloaded and put into place
#	1 if can't get sysinfo record
#	2 if could not initialize FLORDIR
#	3 if could not download patch
#
sub get_patch
{
    my $flordir = initialize_flordir();
    if ($flordir eq "") {
	logerror("Could not initialize FLORDIR... exiting");
	return(2);
    }

    if (download_patch($flordir) != 0) {
	logerror("Could not download patch files... exiting");
	return(3);
    }

    return(0);
}


#
# Here is where the interesting code starts and finishes.
#
exit(main());


__END__

=pod

=head1 NAME

tfgetpatch.pl - Teleflora Patch Downloader


=head1 VERSION

This documenation refers to version: $Revision: 1.1 $


=head1 USAGE

tfgetpatch.pl

tfgetpatch.pl B<--version>

tfgetpatch.pl B<--help>

tfgetpatch.pl B<[--rti] [--rti-tel-file=s] [--rti-cmb-file=s]>

tfgetpatch.pl B<[--daisy] [--daisy-edir-file=s]>


=head1 OPTIONS

=over 4

=item B<--version>

Output the version number of the script and exit.

=item B<--help>

Output a short help message and exit.

=item B<--rti>

Assume the system is an RTI system.

=item B<--daisy>

Assume the system is a Daisy system.

=item B<--rti-tel-file=s>

Optional specification of the filename for the RTI edir patch file
which contains only Teleflora edir files.
If the C<rti-tel-file> command line option is not specified,
the value C<rti-edir-tel-latest.patch> is used.

=item B<--rti-cmb-file=s>

Optional specification of the filename for the RTI edir patch file
which contains both the FTD and Teleflora edir files.
If the C<rti-cmb-file> command line option is not specified,
the value C<rti-edir-cmb-latest.patch> is used.

=item B<--daisy-edir-file=s>

Optional specification of Daisy edir patch file on blob server.
If the C<daisy-edir-file> command line option is not specified,
the value C<daisy-edir-latest.patch> is used.

=back


=head1 DESCRIPTION

The I<tfgetpatch.pl> script gets a set of patch files from the Teleflora "blob" server and
puts them in place, ready to be installed.
Both RTI and Daisy POS systems are supported and under normal circumtances,
the appropriate POS is detected.
An optional command line option may be used to specify the POS.

The script first initalizes the destination directory for the set of patch files.
For RTI this directory is C</usr2/FLORDIR> and for Daisy it is C</d/FLORDIR>.
If the directory does not exist, it is made.
If the directory exists, any files within it are removed.

On RTI systems, the script then determines what type of patch file to download.
For RTI systems, the patch file may be contain only Teleflora edir files or
combined Teleflora and FTD files.
There is only one type of patch file for Daisy systems.

Finally, the script downloads the appropriate set of patch files and
puts them in the destination directory.
At this point, a subsequent operation with an appropriate script can be used
to install the patch.


=head1 FILES

=over 4

=item B</usr2/bbx>

This directory exists only on RTI systems and
is an indication that the RTI POS has been installed.

=item B</usr2/FLORDIR>

This directory only exists on RTI systems and
is the location that edir patch files are placed.

=item B<rti-edir-tel-latest.patch>

The RTI edir patch file which resides on the "blob" server and
contains only Teleflora files.

=item B<rti-edir-cmb-latest.patch>

The RTI edir patch file which resides on the "blob" server and
contains both FTD and Teleflora files.

=item B</d/daisy>

This directory only exists on Daisy systems and
is an indication that the Daisy POS has been installed.

=item B</d/FLORDIR>

This directory only exists on Daisy systems and
is the location that edir patch files are placed.

=item B<daisy-edir-latest.patch>

The Daisy edir patch file which resides on the "blob" server.

=item B<RTI-Patches.log>

Log entries are written to this file;
the logfile is normally located in
the directory B</usr2/bbx/log> for RTI systems and
the directory B</d/daisy/log> for Daisy systems.

=back


=head1 DIAGNOSTICS

=over 4

=item Exit status 0

Successful completion.

=item Exit status 1

In general, there was an issue with the syntax of the command line.

=item Exit status 4

The script could not detect that either the RTI or Daisy POS was installed and 
one or the other must be installed or the script will exit.

=item Exit status 5

The script could not download the appropriate set of patch files from the "blob" server.

=back


=head1 SEE ALSO

applypatch.pl

=cut
