#!/usr/bin/perl
#
# $Revision: 1.5 $
# Copyright 2009 Teleflora
#
# Apply a "patch" to the RTI or "daisy" POS environment.
# Intended for use with "makepatch.pl" script and an Altiris deployment model.
#
use strict;
use warnings;
use Getopt::Long;
use POSIX;
use English;
use Digest::MD5;
use File::Basename;


my @PATCHFILES = ();
my $HELP= 0;
my $VERSION = 0;
my $RESTART = 1;
my $RTI = 0;
my $DAISY = 0;
my $POSDIR = "/usr2/bbx";
my $TIMESTAMP = strftime("%Y%m%d%H%M%S", localtime());
my $thisfile = "";
my $returnval = 0;
my $postpatch_returnval = 0;
my $update_type="Patch";
my $this_is_revision=0;

GetOptions(
	"version" => \$VERSION,
	"restart!" => \$RESTART,
	"help" => \$HELP,
	"rti" => \$RTI,
	"daisy" => \$DAISY,
);


# --version
if($VERSION != 0) {
	print("$0 " . '$Revision: 1.5 $' . "\n");
	exit(0);
}

# --help
if($#ARGV < 0) {
	usage();
	exit(1);
}


# Clean input of "patchfile" names to avoid shell injection
# attacks.
foreach $thisfile(@ARGV) {
	push(@PATCHFILES, validate_input($thisfile));
}


loginfo("");
loginfo("");
loginfo("<===== $0 " . '$Revision: 1.5 $' . " =====>");
loginfo("Applying files: @PATCHFILES");



# Are we root?
if($EUID != 0) {
	usage();
	loginfo("Error: You need Superuser Privileges to run this script.\n");
	loginfo("No Files Applied.");
	loginfo("<===== End $0 " . '$Revision: 1.5 $' . " =====>");
	exit(3);
}




# Make sure we are given valid patch files.
foreach $thisfile (@PATCHFILES) {
	my @patchfiles = ();

	$returnval = get_patch_revision($thisfile);

	# User did not specify a file.
	if (! -f "$thisfile") {
		loginfo("\"$thisfile\" Is Not a File.\n");
		loginfo("No $update_type Applied.");
		loginfo("<===== End $0 " . '$Revision: 1.5 $' . " =====>");
		exit(5);
	}

	# Patchfile is a zero byte file.
	if(-s "$thisfile" == 0) {
		loginfo("\"$thisfile\" Is a Zero Byte File.\n");
		loginfo("No $update_type Applied.");
		loginfo("<===== End $0 " . '$Revision: 1.5 $' . " =====>");
		exit(6);
	}

	# Patchfile is empty.
	@patchfiles = get_patchlist($thisfile);
	if($#patchfiles < 0) {
		loginfo("Patchfile \"$thisfile\" Contains no Files Within it.\n");
		loginfo("No $update_type Applied.");
		loginfo("<===== End $0 " . '$Revision: 1.5 $' . " =====>");
		exit(7);
	}



	# "daisy-blah.patch" files use a different set of verification rules.
	if($thisfile =~ /daisy-.*.\.patch/) {
		$POSDIR = "/d/daisy";
		$DAISY=1; # We have at least one daisy patch.


		# Make sure Daisy Directory is in place.
		if (verify_daisydir($POSDIR) != 0) {
			usage();
			loginfo("Error: Daisy Directory \"$POSDIR\" Not Found.\n");
			loginfo("No Patches Applied.");
			loginfo("<===== End $0 " . '$Revision: 1.5 $' . " =====>");
			exit(2);
		}


	# RTI Patch File Verification.
	} else {
		$POSDIR = "/usr2/bbx";
		$RTI=1; # We have at least one RTI patch.


		# Make sure RTI Directory is in place.
		if (verify_rtidir($POSDIR) != 0) {
			usage();
			loginfo("Error: RTI Directory \"$POSDIR\" Not Found.\n");
			loginfo("No $update_type Applied.");
			loginfo("<===== End $0 " . '$Revision: 1.5 $' . " =====>");
			exit(3);
		}
	}

	# Are we out of disk space for this particular POS?
	$returnval = get_diskused($POSDIR);
	if($returnval == 100) {
		loginfo("Error: Disk is Full: \"$POSDIR\"\n");
		loginfo("No $update_type Applied.");
		loginfo("<===== End $0 " . '$Revision: 1.5 $' . " =====>");
		exit(4);
	}


}



# Clean up post-patch script from previous patches
if(-f "$POSDIR/bin/postpatch.pl") {
	loginfo("Removing older \"$POSDIR/bin/postpatch.pl\" Script");
	mysystem("rm -f $POSDIR/bin/postpatch.pl");
	if(-f "$POSDIR/bin/postpatch.pl") {
		loginfo("Could Not Remove Older \"$POSDIR/bin/postpatch.pl\" script.");
		loginfo("No $update_type Applied.");
		loginfo("<===== End $0 " . '$Revision: 1.5 $' . " =====>");
		exit(8);
	}
}

if (! -d "$POSDIR/backups") {
	mysystem("mkdir $POSDIR/backups");
	mysystem("chmod 777 $POSDIR/backups");
}

# Tick, tick.
# From this point in time, RTI is DOWN. Don't call "exit()" unless
# you clean up and also run "startbbx".
# Kill off background programs *prior to* applying patch.
if( ($RTI != 0)
&&  ($RESTART != 0) ) {
	loginfo("Calling killem");
	mysystem("/usr/bin/sudo $POSDIR/bin/killem");
	killallbbx();
	sleep(5);
}


# Step through each patch file and apply.
$returnval = 0;
foreach $thisfile(@PATCHFILES) {

	next if ("$thisfile" eq "");

	# Is this a daisy, or RTI Patch?
	if($thisfile =~ /daisy-.*.\.patch/) {
		$POSDIR = "/d/daisy";
		loginfo("*** Begin Daisy Patch \"$thisfile\" ***");
	} else {
		$POSDIR = "/usr2/bbx";
		loginfo("*** Begin RTI $update_type \"$thisfile\" ***");
	}


	$returnval = get_patch_revision($thisfile);

	log_patch_info($thisfile);
	$returnval = applypatch($thisfile);
	if($returnval != 0) {
		loginfo("Error: $update_type \"$thisfile\" did not apply. Error=$returnval");
		logerror("$thisfile " . uc($update_type) . " UPDATE STATUS: FAILED");
	}
	loginfo("*** End $update_type \"$thisfile\" ***");
	if($returnval != 0) {
		loginfo("Will not proceed with applying further $update_type.");
		last;
	}
}


# Something bombed out during our patch process. Back everything out.
if($returnval != 0) {
	foreach $thisfile(@PATCHFILES) {
		reverse_patch($thisfile);
	}
}


#
# One of the patches may have created a "postpatch.pl" script.
# If so, run that after all .patch files have been extracted.
# Note that, we assume that postpatch.pl "cleans up" if it fails along the
# way. We are giving the postpatch script the "TIMESTAMP", which tells the
# script where our "backup" files are, we are also giving that script 
# the path to our ".patch" files, so that it knows which files to "reverse out".
#
if($returnval == 0) {

	if(-f "$POSDIR/bin/postpatch.pl") {
		my $md5sum = "";
		my @array = ();
		my $cwd  = getcwd();

		$md5sum = get_md5sum("$POSDIR/bin/postpatch.pl"); #Do this *before* dos2unix.
		mysystem("chown root:root $POSDIR/bin/postpatch.pl");
		mysystem("dos2unix $POSDIR/bin/postpatch.pl");
		mysystem("chmod a+rx $POSDIR/bin/postpatch.pl");
		mysystem("chmod a-w $POSDIR/bin/postpatch.pl");

		loginfo("Running \"$POSDIR/bin/postpatch.pl '$cwd' '$TIMESTAMP' @PATCHFILES\"");
		loginfo("postpatch.pl MD5sum = $md5sum");
		$returnval=mysystem("cd $POSDIR && $POSDIR/bin/postpatch.pl \"$cwd\" \"$TIMESTAMP\" @PATCHFILES");
		$postpatch_returnval = $returnval;
		loginfo("\"$POSDIR/bin/postpatch.pl\" Returned $postpatch_returnval");
	} else {
		if ($RTI != 0) {
			mysystem("$POSDIR/bin/rtiperms.pl $POSDIR");
		}
	}
}


# Restart Background Programs
if( ($RTI != 0)
&&  ($RESTART != 0) ) {
	sleep(2);
	loginfo("Calling startbbx");
	mysystem("/usr/bin/sudo $POSDIR/bin/startbbx");
}

loginfo("<===== End $0 " . '$Revision: 1.5 $' . " =====>");

if ($postpatch_returnval != 0) {
	loginfo("applypatch.pl is terminating with Return value $postpatch_returnval");
	exit($postpatch_returnval);
} else {
	loginfo("applypatch.pl is terminating with Return value $returnval");
	exit($returnval);
}


################################################################
################################################################
################################################################


sub usage
{
	print("Usage:\n");
	print("$0 --version\n");
	print("$0 --help\n");
	print("$0 RTI-x.x.x_A.patch RTI-x.x.x_B.patch RTI-x.x.x_C.patch ...\n");
	print("$0 --no-restart RTI-x.x.x_A.patch RTI-x.x.x_B.patch RTI-x.x.x_C.patch ...\n");
	print("\n");
	print("--no-restart option prevents 'killem/startbbx'.\n");
	print("\n");
}



#
# Are we installing into the right place?
#
sub verify_rtidir
{
	my $rtidir = $_[0];

	if("$rtidir" eq "") {
		return(-1);
	}
	if(! -d "$rtidir") {
		return(-2);
	}

	if( (! -d "$rtidir/bbxd")
	&&  (! -d "$rtidir/bbxp")
	&&  (! -d "$rtidir/bbxh") ) {
		print("Error: Directory is not an RTI Tree. : \"$rtidir\"\n");
		return(-3);
	}


	return(0);
}


sub verify_daisydir
{
	my $daisydir = $_[0];

	if("$daisydir" eq "") {
		return(-1);
	}
	if(! -d "$daisydir") {
		return(-2);
	}

	if(! -f "$daisydir/control.dsy") {
		print("Error: Directory is not a Daisy Tree.: \"$daisydir\"\n");
		return(-3);
	}

	return(0);
}




sub applypatch
{
	my $patchfile = $_[0];
	my @patchfiles = ();
	my $thisfile = "";
	my $returnval = 0;
	my $oldmd5 = "";
	my $newmd5 = "";
	my $old_patch_id = "";
	my $new_patch_id = "";
	my $patch_rti_version = "";
	my $current_version = "12.5.24";
	my $minimum_version = "";


	if("$patchfile" eq "") {
		print("Error: Blank patchfile name specified..\n");
		return(1);
	}
	if(! -f "$patchfile") {
		print("Error: \"$thisfile\": File Not Found.\n");
		return(2);
	}
	if(-s "$patchfile" == 0) {
		print("Error: \"$thisfile\": Zero Byte File.\n");
		return(3);
	}

	if($RTI != 0) {
		loginfo("Getting Current RTI Version");
        	my $rti_ini = "$POSDIR/bbxd/RTI.ini";
		$returnval = mysystem("cp -f $rti_ini ${rti_ini}.bak");
		open INI, "< $rti_ini";
		while(<INI>) {
        		my $line = $_;
			chomp $line;
			next if (! ($line=~m/^VERSION=/));
			$line=~s/^VERSION=//;
			$current_version=$line;
			last;
		}
		close INI;
		loginfo("Current RTI Version $current_version");
		$patch_rti_version=basename($patchfile);
		$patch_rti_version=~s/RTI\-//;
		$patch_rti_version=~s/\.patch//i;
		$patch_rti_version=~s/\.revision//i;
		loginfo("$update_type RTI Version Named $patch_rti_version");
		if ($this_is_revision) {
			mysystem("gzip -dc $patchfile | tar -C \"$POSDIR\" -xvf - bin/postpatch.pl");
			mysystem("gzip -dc $patchfile | tar -C \"$POSDIR\" -xvf - bbxp/revision.bbx");
			open INI, "< $POSDIR/bin/postpatch.pl";
			while(<INI>) {
       				my $line1 = $_;
				chomp $line1;
				next if (! ($line1=~m/^my \$PREVIOUS_VERSION=\"/));
				$line1=~s/^my \$PREVIOUS_VERSION=\"//;
				$line1=~s/\"\;//;
				$minimum_version=$line1;
			}
			close INI;
			loginfo("Minimum version has to be $minimum_version");
		}
	}

	loginfo("Applying $update_type \"$patchfile\" ...");



	# Get a list of what is in our patch file.
	@patchfiles = get_patchlist($patchfile);
	if($#patchfiles < 0) {
		loginfo("Error: No files in patchfile \"$patchfile\".\n");
		return(4);
	}




	# Createa  backup of existing files.
	foreach $thisfile(@patchfiles) {
		if(! -f "$POSDIR/$thisfile") {
			loginfo("\"$POSDIR/$thisfile\" will be a new file from patch. Backup Not Possible.");
			next;
		}

		$oldmd5 = get_md5sum("$POSDIR/$thisfile");
		mysystem("cp -f $POSDIR/$thisfile $POSDIR/backups/" . basename($thisfile) . ".prepatch.$TIMESTAMP");
		if($? != 0) {
			loginfo("Error: File Backup Failed ($?): \"$POSDIR/$thisfile\" -> \"$POSDIR/backups/" . basename($thisfile) . ".prepatch.$TIMESTAMP\"");
			return(5);
		}

		# Verify MD5sum of our "old" and "new" files. They should be the same.
		$newmd5 = get_md5sum("$POSDIR/backups/" . basename($thisfile) . ".prepatch.$TIMESTAMP");
		if("$oldmd5" ne "$newmd5") {
			loginfo("Error: File Backup Failed (MD5): \"$POSDIR/$thisfile\" ($oldmd5) -> \"$POSDIR/backups/" . basename($thisfile) . ".prepatch.$TIMESTAMP\" ($newmd5)");
			return(6);
		}
		loginfo("Backed Up... $POSDIR/$thisfile -> $POSDIR/backups/" . basename($thisfile) . ".prepatch.$TIMESTAMP (MD5=$newmd5)");
	}


	# Extract Patch File
	foreach $thisfile(@patchfiles) {

		# Extract this file.
		mysystem("gzip -dc $patchfile | tar -C \"$POSDIR\" -xvf - $thisfile");
		if($? != 0) {
			loginfo("Error: \"$thisfile\" Patch extract FAILED with error $?. ($patchfile)\n");
			return(7);
		}

		if(! -f "$POSDIR/$thisfile") {
			loginfo("Error: \"$patchfile: $thisfile\" File Not Present after extract.");
			return(8);
		}
		if(-s "$POSDIR/$thisfile" == 0) {
			loginfo("Error: \"$patchfile: $thisfile\" Zero Byte File after extract.");
			return(9);
		}


		$newmd5 = get_md5sum("$POSDIR/$thisfile");
		loginfo("Patched \"$POSDIR/$thisfile\" (New MD5=$newmd5)");
	}


	# Success
	return(0);
}




#
# Back out a previously applied patch, if the patch was even previously applied.
#
sub reverse_patch
{
	my $patchfile = $_[0];
	my $thisfile = "";
	my @patchfiles = ();


	loginfo("!!!!!!!!!!!!!!! Begin Reverse Patch \"$patchfile\" !!!!!!!!!!!!!!!");
	@patchfiles = get_patchlist($patchfile);
	foreach $thisfile(@patchfiles) {

		# No file to be backed out.
		# This could happen if, for example, we were supposed to apply three .patch files, and, the first .patch
		# file hosed before the subsequent two "backup" processes started.
		# In short, this is a fairly harmless situation. If there is no backup file, then, we didn't touch anything
		# to begin with.
		if(! -f "$POSDIR/backups/" . basename($thisfile) . ".prepatch.$TIMESTAMP") {
			loginfo("Warning: $POSDIR/backups/" . basename($thisfile) . ".prepatch.$TIMESTAMP Does Not Exist. Cannot Back-out change.");
			next;
		}

		loginfo("Backing Out $POSDIR/backups/" . basename($thisfile) . ".prepatch.$TIMESTAMP -> $POSDIR/$thisfile");
		mysystem("cp -f $POSDIR/backups/" . basename($thisfile) . ".prepatch.$TIMESTAMP $POSDIR/$thisfile 2> /dev/null");
		if($? != 0) {
			# We're pretty much hosed at this point.
			loginfo("Error: Backout of $POSDIR/backups/" . basename($thisfile) . ".prepatch.$TIMESTAMP $POSDIR/$thisfile Failed ($?)");
		}
	}
	loginfo("!!!!!!!!!!!!!!! End Reverse Patch \"$patchfile\" !!!!!!!!!!!!!!!");

	return(0);
}



#
# Lok in the patch file and get a list of files which will be added.
#
sub get_patchlist
{
	my $patchfile = $_[0];
	my @files = ();


	open(PIPE, "tar -tzvf $patchfile |");
	while(<PIPE>) {
		my $line = $_;
		chomp $line;
		my @tarline = split (/\s+/, $line);
		my $filename = $tarline[5];
		next if($filename =~ /^(\d+\.\d+\.\d+)(\s+)(Patch)$/); # Tar Header
		next if($tarline[0] =~ /^d/); # Directory within the tarball.
		next if(-d "$POSDIR/$filename");
		push(@files, $filename);
	}
	close(PIPE);

	return(@files);
}


#
# Create a log entry indicating what has changed.
#
sub log_patch_info
{
	my $patchfile = $_[0];
	my @patchfiles = ();
	my $patchsize = "";


	loginfo("");
	loginfo("$0  " . '$Revision: 1.5 $');
	loginfo("USER: $ENV{'USER'}");
	loginfo("UID: $UID");
	loginfo("EUID: $EUID");
	loginfo("CWD: " . getcwd());
	loginfo("POSDIR: $POSDIR");

	$patchsize = (-s $patchfile);
	loginfo("$update_type File: $patchfile (" . get_md5sum($patchfile) . ")");
	loginfo("$update_type File Size: $patchsize");
	loginfo("$update_type File MD5: " . get_md5sum($patchfile));
	loginfo("$update_type Timestamp: $TIMESTAMP");

	# Log the actual files in the patch.
	@patchfiles = get_patchlist($patchfile);
	loginfo("$update_type Contents: @patchfiles");
}


#
# Return the percentage of disk being used.
# This gives us a clue as to whether the disk is already "at capacity".
#
sub get_diskused
{
	my $directory = $_[0];
	my $returnval = -1;


	if(! -d "$directory") {
		return(-1);
	}


	open(PIPE, "df -h $directory |");
	while(<PIPE>) {
		next until(/(\s+)(\d+)(\%)(\s+)/);
		$returnval = $2;
		last;
	}
	close(PIPE);

	return($returnval);
}


sub get_md5sum
{
	my $filename = $_[0];
	my $ctx;
	my $md5sum = "";

	if(! -f "$filename") {
		return("");
	}

	open(FILE, "< $filename");
	$ctx = Digest::MD5->new;
	$ctx->addfile(*FILE);
	$md5sum = $ctx->hexdigest;
	close(FILE);

	return($md5sum);
}


sub loginfo
{
	my $message = $_[0];
	my $logtime = strftime("%Y-%m-%d %H:%M:%S", localtime());

	chomp($message);
	open LOGFILE, ">> $POSDIR/log/RTI-Patches.log";
		print LOGFILE "$logtime ($0-$$) <I> $message\r\n";
	close LOGFILE;

	return "";
}


sub logerror
{
	my $message = $_[0];
	my $logtime = strftime("%Y-%m-%d %H:%M:%S", localtime());

	chomp($message);
	open LOGFILE, ">> $POSDIR/log/RTI-Patches.log";
		print LOGFILE "$logtime ($0-$$) <E> $message\r\n";
	close LOGFILE;

	return "";
}


# PCI 6.5.6
# "some string; `cat /etc/passwd`" -> "some string cat /etc/passwd"
# "`echo $ENV; $(sudo ls)`" -> "echo ENV sudo ls"
# etc.
sub validate_input
{
	my $var = $_[0];
	my $temp = "";

	if(! $var) {
		return "";
	}

	$temp = $var;

	# "`bad command`" -> "bad command"
	$temp =~ s/\`//g;

	# "$(bad command)" -> "bad command"
	$temp =~ s/(\$\()(.*.)(\))/$2/g;


	# "stuff ; bad command" -> "stuff bad command"
	$temp =~ s/\;//g;

	# "stuff && bad command" -> "stuff bad command"
	$temp =~ s/\&//g;

	# "stuff | bad command" -> "stuff bad command"
	$temp =~ s/\|//g;

	# "stuff > bad command" -> "stuff bad command"
	$temp =~ s/\>//g;

	# "stuff < bad command" -> "stuff bad command"
	$temp =~ s/\<//g;

	# Filter non printables
	$temp =~ s/[[:cntrl:]]//g;


	return($temp);
}

#
# Subroutine to run system commands.  This logs an entry.
#
sub mysystem
{
        my $command = $_[0];
        my $returnval = "";
	my $LOG_FILE = "$POSDIR/log/RTI-Patches.log";

        system($command . ">> $LOG_FILE 2>> /dev/null");
        $returnval = WEXITSTATUS($?);
        loginfo($command . " Returned $returnval");

        return ($returnval)
}

## Assigns whether file is patch or revision.
sub get_patch_revision
{
	$update_type="Patch";
	$this_is_revision=0;
	if ($_[0]=~m/revision/) {
		$update_type="Revision";
		$this_is_revision=1;
	}
}


sub killallbbx
{
open(PIPE, "ps wwaux |");
        while(<PIPE>) {
                chomp;
                next until(/RTISTART/);

                my @array = split(/\s+/);
                my $username = $array[0];
                my $pid = $array[1];
                my $terminal = $array[6];

                system("sudo kill -HUP $pid");
                sleep(1);
                system("sudo kill -TERM $pid 2> /dev/null");
                sleep(1);
                system("sudo kill -KILL $pid 2> /dev/null");

	}
        close(PIPE);
}
